<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Resources\User\UserResource;
use App\sys\ApiResponse;
use App\sys\Repository\General\CurrencyRepository;
use App\sys\Services\Auth\AuthService;
use Illuminate\Http\Request;

class AuthController extends Controller
{
    use ApiResponse;

    protected $authservice;

    protected $currencyrepo;

    public function __construct(AuthService $authservice, CurrencyRepository $currencyrepo)
    {
        $this->authservice = $authservice;
        $this->currencyrepo = $currencyrepo;
    }

    public function Login(Request $request)
    {
        $result = $this->authservice->login($request->only('email', 'password'));
        if (! $result) {
            $errors = $this->authservice->errors();

            return $this->apiResponse(401, 'fail', $errors, null);
        }
        $defaultcurrency = $this->currencyrepo->getDefault();
        return $this->apiResponse(200, 'success', null, [
            'access_token' => 'Bearer '.$result['access_token'],
            'refresh_token' => $result['refresh_token'],
            'data' => new UserResource($result['user']),
            'default_currency' => $defaultcurrency,
        ]);
    }

    public function logout()
    {
        $result = $this->authservice->logout();

        if (! $result) {
            $errors = $this->authservice->errors();

            return $this->apiResponse(500, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function refresh(Request $request)
    {
        $refreshToken = $request->refresh_token;

        if (! $refreshToken) {
            return $this->apiResponse(400, 'fail', ['token' => ['Refresh token is required']], null);
        }

        $tokenData = $this->authservice->refreshToken($refreshToken);

        if (! $tokenData) {
            $errors = $this->authservice->errors();

            // Check for specific error messages to return appropriate status code
            if (isset($errors['token']) && strpos($errors['token'][0], 'used already') !== false) {
                return $this->apiResponse(401, 'fail', $errors, null); // Security risk
            }

            return $this->apiResponse(401, 'fail', $errors, null);
        }

        // رجع التوكن الجديد بنفس شكل الـ format
        return $this->apiResponse(200, 'success', null, [
            'access_token' => 'Bearer '.$tokenData['access_token'],
            'refresh_token' => $tokenData['refresh_token'],
            'token_type' => 'Bearer',
            'expires_in' => $tokenData['expires_in'],
        ]);
    }

    public function changePassword(Request $request)
    {
        $data = $request->only(
            'old_password',
            'new_password',
            'new_password_confirmation'
        );

        $success = $this->authservice->changePassword($data);

        if (! $success) {
            $errors = $this->authservice->errors();

            return $this->apiResponse(500, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function me()
    {

        $user = auth('api')->user();
        $defaultcurrency = $this->currencyrepo->getDefault();

        $user->load(
            'role',
            'groups',
            'services',
            'accommodations',
            'companies',
            'account_mappings.currency',
            'account_mappings.commissionAccount',
            'account_mappings.walletAccount'
        );

        return $this->apiResponse(200, 'success', null, [
            'data' => new UserResource($user),
            'default_currency' => $defaultcurrency,
        ]);
    }
}
