<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\AccommodationReservationCollection;
use App\Http\Resources\Profile\AccommodationReservationResource;
use App\sys\ApiResponse;
use App\sys\Services\Profile\AccommodationReservationService;
use Illuminate\Http\Request;

class AccommodationReservationController extends Controller
{
    use ApiResponse;

    protected AccommodationReservationService $reservationService;

    public function __construct(AccommodationReservationService $reservationService)
    {
        $this->reservationService = $reservationService;
    }

    public function index(Request $request)
    {
        $items = $this->reservationService->getPaginated($request->all());

        return $this->apiResponse(200, 'success', null, new AccommodationReservationCollection($items));
    }

    public function show(int $id)
    {
        $item = $this->reservationService->getById($id);
        if (! $item) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationResource($item));
    }

    public function store(Request $request)
    {
        $response = $this->reservationService->create($request->all());
        if (! $response) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->reservationService->update($request->all());
        if (! $response) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->reservationService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Accommodation reservations deleted successfully', null, null);
        }
        $errors = $this->reservationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getByAccommodationType(string $type)
    {
        $data = $this->reservationService->getByAccommodationType($type);
        if (! $data) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationCollection($data));
    }

    public function getByAccommodationTypeAndProfile(string $type, int $profileId)
    {
        $data = $this->reservationService->getByAccommodationTypeAndProfile($type, $profileId);
        if (! $data) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationCollection($data));
    }

    public function reservationAttachments(Request $request)
    {
        $data = $this->reservationService->reservationAttachments($request->all());
        if (! $data) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new \App\Http\Resources\Profile\AccommodationReservationAttachmentResource($data));
    }

    public function deleteAttachment(Request $request)
    {
        $result = $this->reservationService->deleteAttachment($request->all());

        if (! $result) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function attachmentsByProfile(int $profileId)
    {
        $items = $this->reservationService->getAttachmentsByProfileId($profileId);
        if (! $items) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, \App\Http\Resources\Profile\AccommodationReservationAttachmentResource::collection($items));
    }

    public function attachmentsByReservation(int $reservationId)
    {
        $items = $this->reservationService->getAttachmentsByReservationId($reservationId);
        if (! $items) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, \App\Http\Resources\Profile\AccommodationReservationAttachmentResource::collection($items));
    }

    public function status(Request $request)
    {
        $response = $this->reservationService->updateStatus($request->all());
        if (! $response) {
            $errors = $this->reservationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationResource($response));
    }
}
