<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\CustomerProfileResource;
use App\sys\ApiResponse;
use App\sys\Services\Profile\CustomerProfilesService;
use Illuminate\Http\Request;

class CustomerProfilesController extends Controller
{
    use ApiResponse;

    protected CustomerProfilesService $customerProfilesService;

    public function __construct(CustomerProfilesService $customerProfilesService)
    {
        $this->customerProfilesService = $customerProfilesService;
    }

    public function getCustomerAccountStatement(Request $request)
    {
        $result = $this->customerProfilesService->getCustomerAccountStatement($request->all());
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }

    /**
     * Get customer profiles with costs and payments by currency
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerProfiles($customerId)
    {
        $result = $this->customerProfilesService->getCustomerProfiles($customerId);

        // Check if validation failed (result is false)
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        // Return success with data (even if empty array)
        return $this->apiResponse(200, 'success', null, [
            'profiles' => CustomerProfileResource::collection($result),
            'total_profiles' => count($result),
        ]);
    }

    /**
     * Get specific profile details with costs and payments by currency
     *
     * @param  int  $profileId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProfileDetails($profileId)
    {
        $result = $this->customerProfilesService->getProfileDetails($profileId);

        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, [
            'profile_id' => (int) $profileId,
            'currencies' => $result,
            'total_currencies' => count($result),
        ]);
    }

    /**
     * Get customer financial summary by currency (aggregated across all profiles)
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerFinancialSummary($customerId)
    {

        $result = $this->customerProfilesService->getCustomerFinancialSummary($customerId);

        // Check if validation failed (result is false)
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        // Return success with data (even if empty array)
        return $this->apiResponse(200, 'success', null, [
            'currencies' => $result,
            'summary' => $this->getFinancialSummary($result),
        ]);
    }

    /**
     * Customer invoice balances grouped by customer (optionally by company)
     */
    public function getCustomerInvoiceBalances(Request $request)
    {
        $result = $this->customerProfilesService->getCustomerInvoiceBalances($request->get('company_id'));
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }

    /**
     * Endpoint: استعلام كشف محفظة العميل
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCustomerWalletStatement(Request $request)
    {
        $data = $request->all();
        $data['type_optional'] = 'wallet';
        $result = $this->customerProfilesService->getCustomerAccountStatement($data);
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }

    /**
     * كشف حركات محفظة العميل فقط
     */
    public function getCustomerWalletTransactions(Request $request)
    {

        $result = $this->customerProfilesService->getCustomerWalletTransactions($request->all());
        if ($result != false) {
            return $this->apiResponse(200, 'success', null, $result);
        }

        $errors = $this->customerProfilesService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);

    }

    /**
     * رصيد المحفظة فقط Wallet Balance
     */
    public function getCustomerWalletBalanceV2(Request $request)
    {
        if ($result = $this->customerProfilesService->getCustomerWalletBalance($request->all())) {
            return $this->apiResponse(200, 'success', null, $result);
        }

        $errors = $this->customerProfilesService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);

    }

    /**
     * ملخص محافظ العملاء: اسم العميل، إجمالي مدين، إجمالي دائن، والفرق
     * Params (POST JSON): customer_ids (array|null), from (date|null), to (date|null)
     */
    public function getCustomersWalletSummary(Request $request)
    {
        $result = $this->customerProfilesService->getCustomersWalletSummary($request->all());
        if ($result === false) {
            $errors = $this->customerProfilesService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, $result);
    }

    /**
     * Get financial summary totals converted to default currency
     *
     * @param  array  $currencies
     * @return array
     */
    private function getFinancialSummary($currencies)
    {
        $defaultCurrency = \App\Models\General\Currency::where('is_default', 1)->first();

        if (! $defaultCurrency) {
            // Fallback to first currency if no default found
            $defaultCurrency = collect($currencies)->first();
            if (! $defaultCurrency) {
                return [
                    'total_cost' => 0,
                    'total_paid' => 0,
                    'total_remaining' => 0,
                    'payment_percentage' => 0,
                    'default_currency' => null,
                ];
            }
        }

        $totalCostInDefault = 0;
        $totalPaidInDefault = 0;
        $totalRemainingInDefault = 0;

        foreach ($currencies as $currency) {
            $exchangeRate = $currency['currency_id'] == $defaultCurrency->id ? 1 : $this->getExchangeRate($currency['currency_id'], $defaultCurrency->id);

            $totalCostInDefault += $currency['total_cost'] * $exchangeRate;
            $totalPaidInDefault += $currency['total_paid'] * $exchangeRate;
            $totalRemainingInDefault += $currency['remaining'] * $exchangeRate;
        }

        return [
            'total_cost' => round($totalCostInDefault, 2),
            'total_paid' => round($totalPaidInDefault, 2),
            'total_remaining' => round($totalRemainingInDefault, 2),
            'payment_percentage' => $totalCostInDefault > 0 ? round(($totalPaidInDefault / $totalCostInDefault) * 100, 2) : 0,
            'default_currency' => [
                'id' => $defaultCurrency->id,
                'name' => $defaultCurrency->name,
                'code' => $defaultCurrency->code,
                'symbol' => $defaultCurrency->symbol,
            ],
        ];
    }

    public function wallet(Request $request)
    {
        if ($const = $this->customerProfilesService->wallet($request->all())) {
            return $this->apiResponse(200, 'success', null, $const);
        }

        $errors = $this->customerProfilesService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    /**
     * Get exchange rate between two currencies
     *
     * @param  int  $fromCurrencyId
     * @param  int  $toCurrencyId
     * @return float
     */
    private function getExchangeRate($fromCurrencyId, $toCurrencyId)
    {
        $fromCurrency = \App\Models\General\Currency::find($fromCurrencyId);
        $toCurrency = \App\Models\General\Currency::find($toCurrencyId);

        if (! $fromCurrency || ! $toCurrency) {
            return 1; // Fallback to 1 if currencies not found
        }

        // Simple exchange rate calculation
        // You might need to implement a more sophisticated exchange rate system
        return $fromCurrency->exchange_rate / $toCurrency->exchange_rate;
    }
}
