<?php

namespace App\Http\Resources\Invoice;

use App\sys\Services\Invoice\InvoiceServicesTypeConfig;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class InvoiceServicesResource extends JsonResource
{
    protected $serviceType;

    public function __construct($resource, ?string $serviceType = null)
    {
        parent::__construct($resource);
        $this->serviceType = $serviceType ?? $resource->travel_tourism_type ?? 'accommodation';
    }

    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $resourceFields = InvoiceServicesTypeConfig::getResourceFields($this->serviceType);
        $relationships = InvoiceServicesTypeConfig::getRelationships($this->serviceType);

        $data = [];

        // Add basic fields (include even if null)
        foreach ($resourceFields as $field) {
            $data[$field] = $this->$field;
        }

        // Special handling for flights type
        if ($this->serviceType === 'flights') {
            return $this->formatFlightsData($data, $relationships);
        }

        // Add relationships
        foreach ($relationships as $relationship) {
            // For segments and airlineTickets, always return array (even if empty or not loaded)
            if (in_array($relationship, ['segments', 'airlineTickets'])) {
                if ($this->relationLoaded($relationship)) {
                    $collection = $this->$relationship ?? collect([]);
                    if ($relationship === 'segments') {
                        $data[$relationship] = \App\Http\Resources\Invoice\SegmentsResource::collection($collection);
                    } elseif ($relationship === 'airlineTickets') {
                        $data[$relationship] = \App\Http\Resources\Invoice\AirlineTicketsResource::collection($collection);
                    } else {
                        $data[$relationship] = $collection;
                    }
                } else {
                    // Load relationship if not loaded
                    $this->resource->loadMissing($relationship);
                    $collection = $this->resource->$relationship ?? collect([]);
                    if ($relationship === 'segments') {
                        $data[$relationship] = \App\Http\Resources\Invoice\SegmentsResource::collection($collection);
                    } elseif ($relationship === 'airlineTickets') {
                        $data[$relationship] = \App\Http\Resources\Invoice\AirlineTicketsResource::collection($collection);
                    } else {
                        $data[$relationship] = $collection;
                    }
                }
            } else {
                // For other relationships, use whenLoaded
                if ($this->relationLoaded($relationship)) {
                    $data[$relationship] = $this->whenLoaded($relationship);
                } else {
                    // Load relationship if not loaded
                    $this->resource->loadMissing($relationship);
                    $data[$relationship] = $this->whenLoaded($relationship);
                }
            }
        }

        return $data;
    }

    /**
     * Format flights data to match the input format with additional relationship data
     */
    private function formatFlightsData(array $data, array $relationships): array
    {
        // Load relationships if not loaded
        $this->resource->loadMissing(['segments.originAirport', 'segments.destinationAirport', 'segments.flightCompany', 'airlineTickets.fromAirport', 'airlineTickets.toAirport', 'airlineTickets.fromAirline', 'airlineTickets.toAirline']);

        // Format segments
        $segments = [];
        foreach ($this->resource->segments ?? [] as $segment) {
            $segmentData = [
                'origin_airport' => $segment->origin_airport,
                'destination_airport' => $segment->destination_airport,
                'date' => $segment->date?->format('Y-m-d'),
                'time' => $segment->time,
                'type' => $segment->type,
                'flight_number' => $segment->flight_number,
                'flight_company' => $segment->flight_company,
            ];

            // Add relationship data
            if ($segment->relationLoaded('originAirport') && $segment->originAirport) {
                $segmentData['origin_airport_data'] = $segment->originAirport;
            }
            if ($segment->relationLoaded('destinationAirport') && $segment->destinationAirport) {
                $segmentData['destination_airport_data'] = $segment->destinationAirport;
            }
            if ($segment->relationLoaded('flightCompany') && $segment->flightCompany) {
                $segmentData['flight_company_data'] = $segment->flightCompany;
            }

            $segments[] = $segmentData;
        }
        $data['segments'] = $segments;

        // ضرائب: إرجاع tax_rate_id كـ IDs فقط إذا كان محملاً
        if ($this->relationLoaded('taxes')) {
            $data['tax_rate_id'] = $this->taxes->pluck('tax_rate_id')->filter()->values();
        } else {
            $data['tax_rate_id'] = [];
        }

        // Rebuild pax structure from airlineTickets
        $airlineTickets = $this->resource->airlineTickets ?? collect([]);

        // نعيد أيضاً airlineTickets كما هي (للوصول المباشر إلى chair_num / flight_num / ticket_num)
        $data['airlineTickets'] = AirlineTicketsResource::collection($airlineTickets);

        // Group tickets by passenger type and direction
        $packsPrice = [
            'adult' => ['count' => 0, 'cost' => 0, 'tax' => 0, 'earned_discount' => 0, 'granted_discount' => 0, 'total_emd' => 0, 'total_cost' => 0],
            'child' => ['count' => 0, 'cost' => 0, 'tax' => 0, 'earned_discount' => 0, 'granted_discount' => 0, 'total_emd' => 0, 'total_cost' => 0],
            'infant' => ['count' => 0, 'cost' => 0, 'tax' => 0, 'earned_discount' => 0, 'granted_discount' => 0, 'total_emd' => 0, 'total_cost' => 0],
        ];

        $packsDetails = [];
        $passengerGroups = [];
        $countedPassengers = []; // Track counted passengers to avoid double counting

        foreach ($airlineTickets as $ticket) {
            $passengerType = strtolower($ticket->passenger_type ?? 'adult');
            $directionType = $ticket->go_back ?? 'go';

            // Count unique passengers (group by name and type, ignoring direction)
            $passengerKey = $ticket->passenger_name.'|'.$passengerType;
            if (! isset($passengerGroups[$passengerKey])) {
                $nameParts = $this->extractNameParts($ticket->passenger_name);
                $passengerGroups[$passengerKey] = [
                    'title' => $ticket->title,
                    'type' => ucfirst($passengerType),
                    'first_name' => $nameParts['first_name'],
                    'last_name' => $nameParts['last_name'],
                    'passport_number' => $ticket->passport_num,
                    'ticket_number' => $ticket->ticket_num,
                    'emd' => $ticket->ticket_emd === 'NONE' ? null : $ticket->ticket_emd,
                    'chair_number' => $ticket->chair_num,
                    'notes' => $ticket->note,
                ];
            }

            // Update packs_price totals - aggregate all tickets for this passenger type
            if (isset($packsPrice[$passengerType])) {
                // Count unique passengers only once (when processing 'go' direction)
                if ($directionType === 'go' && ! isset($countedPassengers[$passengerKey])) {
                    $packsPrice[$passengerType]['count']++;
                    $countedPassengers[$passengerKey] = true;
                }

                // Sum all costs (for both go and back if round trip)
                $packsPrice[$passengerType]['cost'] += $ticket->fare_price ?? 0;
                $packsPrice[$passengerType]['tax'] += $ticket->tax ?? 0;
                $packsPrice[$passengerType]['earned_discount'] += $ticket->earned_discount ?? 0;
                $packsPrice[$passengerType]['granted_discount'] += $ticket->granted_discount ?? 0;
                $packsPrice[$passengerType]['total_emd'] += $ticket->emd_price ?? 0;
                $packsPrice[$passengerType]['total_cost'] += $ticket->total_price ?? 0;
            }

            // Build packs_details with direction - create entry for each ticket
            $detailKey = $passengerKey.'|'.$directionType.'|'.$ticket->id;
            $detail = $passengerGroups[$passengerKey];
            $detail['direction_type'] = $directionType;
            // إعادة نفس الحقول المطلوبة في الـ request
            $detail['ticket_number'] = $ticket->ticket_num;
            $detail['chair_number'] = $ticket->chair_num;
            $detail['flight_num'] = $ticket->flight_num;

            // Add relationship data
            if ($ticket->relationLoaded('fromAirport') && $ticket->fromAirport) {
                $detail['from_airport_data'] = $ticket->fromAirport;
            }
            if ($ticket->relationLoaded('toAirport') && $ticket->toAirport) {
                $detail['to_airport_data'] = $ticket->toAirport;
            }
            if ($ticket->relationLoaded('fromAirline') && $ticket->fromAirline) {
                $detail['from_airline_data'] = $ticket->fromAirline;
            }
            if ($ticket->relationLoaded('toAirline') && $ticket->toAirline) {
                $detail['to_airline_data'] = $ticket->toAirline;
            }

            $packsDetails[$detailKey] = $detail;
        }

        // Convert packs_details to array
        $data['pax'] = [
            'packs_price' => $packsPrice,
            'packs_details' => array_values($packsDetails),
        ];

        // Add other relationships
        foreach ($relationships as $relationship) {
            if (! in_array($relationship, ['segments', 'airlineTickets'])) {
                if ($this->relationLoaded($relationship)) {
                    $data[$relationship] = $this->whenLoaded($relationship);
                } else {
                    $this->resource->loadMissing($relationship);
                    $data[$relationship] = $this->whenLoaded($relationship);
                }
            }
        }

        return $data;
    }

    /**
     * Extract first name and last name from passenger_name
     * Format: "title first_name last_name" or "first_name last_name" or "first_name"
     */
    private function extractNameParts(?string $passengerName): array
    {
        if (! $passengerName) {
            return ['first_name' => '', 'last_name' => ''];
        }

        $parts = array_filter(explode(' ', trim($passengerName)));
        $parts = array_values($parts); // Re-index array

        if (count($parts) === 0) {
            return ['first_name' => '', 'last_name' => ''];
        }

        if (count($parts) === 1) {
            return ['first_name' => $parts[0], 'last_name' => ''];
        }

        if (count($parts) === 2) {
            return ['first_name' => $parts[0], 'last_name' => $parts[1]];
        }

        // 3 or more parts: assume first part is title, second is first_name, rest is last_name
        return [
            'first_name' => $parts[1] ?? '',
            'last_name' => implode(' ', array_slice($parts, 2)),
        ];
    }
}
