<?php

namespace App\Observers;

use App\Models\invoice\InvoiceServices;
use App\Models\Suppliers\Suppliers;
use App\sys\Enums\GroupEnum;
use App\sys\Services\Notification\NotificationService;

class InvoiceServicesObserver
{
    protected NotificationService $notificationService;

    /**
     * المجموعات التي يجب إرسال إشعارات التحديثات المالية والموردين لها
     */
    protected const MANAGEMENT_GROUPS = [
        GroupEnum::ADMIN,
        GroupEnum::HANDLING_MANAGER,
        GroupEnum::ACCOUNTANT_MANAGER,
    ];

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Handle the InvoiceServices "created" event.
     */
    public function created(InvoiceServices $invoiceService): void
    {
        // إرسال الإشعار فقط إذا لم تكن الخدمة من نوع accommodation
        if ($invoiceService->travel_tourism_type === 'accommodation') {
            return;
        }

        // جلب اسم الخدمة
        $serviceName = $invoiceService->service?->title ?? 'غير محدد';

        // إرسال إشعار لجميع الـ Admins عند إنشاء InvoiceService جديد
        $this->notificationService->sendToGroup(
            groupId: GroupEnum::ADMIN->value,
            type: 'invoice_update',
            title: 'تم إنشاء حجز خدمه جديد ',
            message: "تم إنشاء حجز خدمه جديد برقم {$invoiceService->id} - الخدمة: {$serviceName}",
            modelName: 'InvoiceServices',
            modelId: $invoiceService->id,
            data: [
                'invoice_service_id' => $invoiceService->id,
                'service_id' => $invoiceService->service_id,
                'service_name' => $serviceName,
                'travel_tourism_type' => $invoiceService->travel_tourism_type,
                'action' => 'created',
            ]
        );

        // إرسال إشعار لجميع الـ Handling Managers عند إنشاء InvoiceService جديد
        $this->notificationService->sendToGroup(
            groupId: GroupEnum::HANDLING_MANAGER->value,
            type: 'invoice_update',
            title: 'تم إنشاء حجز خدمه جديد ',
            message: "تم إنشاء حجز خدمه جديد برقم {$invoiceService->id} - الخدمة: {$serviceName}",
            modelName: 'InvoiceServices',
            modelId: $invoiceService->id,
            data: [
                'invoice_service_id' => $invoiceService->id,
                'service_id' => $invoiceService->service_id,
                'service_name' => $serviceName,
                'travel_tourism_type' => $invoiceService->travel_tourism_type,
                'action' => 'created',
            ]
        );
    }

    public function updated(InvoiceServices $invoiceService): void
    {
        if ($invoiceService->travel_tourism_type === 'accommodation') {
            return;
        }

        $serviceName = $invoiceService->service?->title ?? 'غير محدد';

        // التحقق من تغيير executive_id
        if ($invoiceService->wasChanged('executive_id') && $invoiceService->executive_id) {
            $executiveName = $invoiceService->executive?->name ?? 'غير محدد';

            // إرسال إشعار للجروبين: تم اختيار [اسم المستخدم] لقيام بالخدمة [اسم الخدمة]
            $this->notificationService->sendToGroup(
                groupId: GroupEnum::ADMIN->value,
                type: 'invoice_update',
                title: 'تم اختيار مسؤول تنفيذ للخدمة',
                message: "تم اختيار {$executiveName} لقيام بالخدمة {$serviceName}",
                modelName: 'InvoiceServices',
                modelId: $invoiceService->id,
                data: [
                    'invoice_service_id' => $invoiceService->id,
                    'service_id' => $invoiceService->service_id,
                    'service_name' => $serviceName,
                    'executive_id' => $invoiceService->executive_id,
                    'executive_name' => $executiveName,
                    'travel_tourism_type' => $invoiceService->travel_tourism_type,
                    'action' => 'executive_assigned',
                ]
            );

            $this->notificationService->sendToGroup(
                groupId: GroupEnum::HANDLING_MANAGER->value,
                type: 'invoice_update',
                title: 'تم اختيار مسؤول تنفيذ للخدمة',
                message: "تم اختيار {$executiveName} لقيام بالخدمة {$serviceName}",
                modelName: 'InvoiceServices',
                modelId: $invoiceService->id,
                data: [
                    'invoice_service_id' => $invoiceService->id,
                    'service_id' => $invoiceService->service_id,
                    'service_name' => $serviceName,
                    'executive_id' => $invoiceService->executive_id,
                    'executive_name' => $executiveName,
                    'travel_tourism_type' => $invoiceService->travel_tourism_type,
                    'action' => 'executive_assigned',
                ]
            );

            // إرسال إشعار للمستخدم المختار: تم اختيارك في الخدمة [اسم الخدمة]
            $this->notificationService->sendToUsers(
                userIds: [$invoiceService->executive_id],
                type: 'invoice_update',
                title: 'تم اختيارك في خدمة',
                message: "تم اختيارك في الخدمة {$serviceName}",
                modelName: 'InvoiceServices',
                modelId: $invoiceService->id,
                data: [
                    'invoice_service_id' => $invoiceService->id,
                    'service_id' => $invoiceService->service_id,
                    'service_name' => $serviceName,
                    'executive_id' => $invoiceService->executive_id,
                    'travel_tourism_type' => $invoiceService->travel_tourism_type,
                    'action' => 'executive_assigned',
                ]
            );
        }

        // التحقق من تغيير grand_total
        if ($invoiceService->wasChanged('grand_total')) {
            $oldGrandTotal = (float) ($invoiceService->getOriginal('grand_total') ?? 0);
            $newGrandTotal = (float) ($invoiceService->grand_total ?? 0);

            // تحديد نوع الرسالة بناءً على القيمة القديمة والجديدة
            if ($oldGrandTotal == 0 && $newGrandTotal > 0) {
                $title = 'تم تحديد سعر الخدمة';
                $message = "تم تحديد سعر الخدمة {$serviceName} - السعر: {$newGrandTotal}";
                $action = 'price_set';
            } else {
                $title = 'تم تغيير سعر الخدمة';
                $message = "تم تغيير سعر الخدمة {$serviceName} من {$oldGrandTotal} إلى {$newGrandTotal}";
                $action = 'price_changed';
            }

            // إرسال إشعار للجروبين الثلاثة
            $this->sendToManagementGroups(
                invoiceService: $invoiceService,
                serviceName: $serviceName,
                title: $title,
                message: $message,
                data: [
                    'old_grand_total' => $oldGrandTotal,
                    'new_grand_total' => $newGrandTotal,
                    'action' => $action,
                ]
            );
        }

        // التحقق من تغيير supplier_id
        if ($invoiceService->wasChanged('supplier_id')) {
            $oldSupplierId = $invoiceService->getOriginal('supplier_id');
            $newSupplierId = $invoiceService->supplier_id;
            $supplierName = $invoiceService->supplier?->name ?? 'غير محدد';

            // تحديد نوع الرسالة بناءً على القيمة القديمة والجديدة
            if (empty($oldSupplierId) && ! empty($newSupplierId)) {
                $title = 'تم تحديد Supplier';
                $message = "تم تحديد Supplier للخدمة {$serviceName} - {$supplierName}";
                $action = 'supplier_set';
            } else {
                // جلب اسم الـ supplier القديم
                $oldSupplierName = 'غير محدد';
                if ($oldSupplierId) {
                    $oldSupplier = Suppliers::find($oldSupplierId);
                    $oldSupplierName = $oldSupplier?->name ?? 'غير محدد';
                }
                $title = 'تم تغيير الـ Supplier';
                $message = "تم تغيير الـ Supplier للخدمة {$serviceName} من {$oldSupplierName} إلى {$supplierName}";
                $action = 'supplier_changed';
            }

            // إرسال إشعار للجروبين الثلاثة
            $this->sendToManagementGroups(
                invoiceService: $invoiceService,
                serviceName: $serviceName,
                title: $title,
                message: $message,
                data: [
                    'old_supplier_id' => $oldSupplierId,
                    'new_supplier_id' => $newSupplierId,
                    'supplier_name' => $supplierName,
                    'action' => $action,
                ]
            );
        }
    }

    /**
     * إرسال إشعار للجروبين الإدارية (Admin, Handling Manager, Accountant Manager)
     */
    protected function sendToManagementGroups(
        InvoiceServices $invoiceService,
        string $serviceName,
        string $title,
        string $message,
        array $additionalData = []
    ): void {
        $baseData = [
            'invoice_service_id' => $invoiceService->id,
            'service_id' => $invoiceService->service_id,
            'service_name' => $serviceName,
            'travel_tourism_type' => $invoiceService->travel_tourism_type,
        ];

        $data = array_merge($baseData, $additionalData);

        foreach (self::MANAGEMENT_GROUPS as $group) {
            $this->notificationService->sendToGroup(
                groupId: $group->value,
                type: 'invoice_update',
                title: $title,
                message: $message,
                modelName: 'InvoiceServices',
                modelId: $invoiceService->id,
                data: $data,
                pushChannel: 'group'
            );
        }
    }
}
