<?php

namespace App\sys\Repository\General;

use App\Models\General\WorkPeriod;

class WorkPeriodRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'period',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $period = request('name', null);
        $limit = request('limit', 15);

        return WorkPeriod::when($period, function ($q, $period) {
            $q->whereHas('currentTranslation', function ($q) use ($period) {
                $q->where('period', 'LIKE', "%$period%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'period') {
                $query->leftJoin('ge_work_period_translation', function ($join) {
                    $join->on('ge_work_period.id', '=', 'ge_work_period_translation.work_period_id')
                        ->where('ge_work_period_translation.lang_id', app('lang_id'));
                })->orderBy('ge_work_period_translation.period', $order)->select('ge_work_period.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return WorkPeriod::with('currentTranslation')->find($id);
    }

    public function create(array $data)
    {
        $item = new WorkPeriod;
        $item->period = $data['name'];
        $item->is_active = $data['active'];
        $item->save();

        return $item;
    }

    public function update(WorkPeriod $item, array $data)
    {
        $item->period = $data['name'] ?? $item->period;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->save();

        return $item;
    }

    public function delete(WorkPeriod $item)
    {
        return $item->delete();
    }

    public function del(array $ids)
    {
        return WorkPeriod::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return WorkPeriod::with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return WorkPeriod::with('translations')->find($id);
    }

    public function getAllChanges($workPeriod)
    {
        return $workPeriod->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
