<?php

namespace App\sys\Repository\User;

use App\Models\User;
use App\Models\UserAccountMapping;
use Illuminate\Support\Facades\DB;

class UserRepository
{
    public function __construct()
    {
        //
    }

    protected $columns = [
        'id' => 'id',
        'name' => 'name',
        'email' => 'email',
        'phone' => 'phone',
        'created_at' => 'created_at',
    ];

    public function getPaginated()
    {

        $column = request('sort_by', null);
        $order = request('sort_order') ?: 'desc';
        $limit = request('limit', 15);

        $name = request('name', null);
        $email = request('email', null);
        $phone = request('phone', null);
        $roleId = request('role_id', null);
        $isActive = request('is_active', null);

        $groupId = request('group_id', null);
        $serviceId = request('service_id', null);
        $accommodationId = request('accommodation_id', null);
        $companyId = request('company_id', null);

        $currencyId = request('currency_id', null);

        $query = User::query()->with([
            'role',
            'groups',
            'services',
            'accommodations',
            'companies',
            'account_mappings.currency',
            'account_mappings.commissionAccount',
            'account_mappings.walletAccount',
            'account_mappings.supplierAccount',
        ]);

        $query->when($name, function ($q, $name) {
            $q->where('name', 'LIKE', "%$name%");
        })
            ->when($email, function ($q, $email) {
                $q->where('email', 'LIKE', "%$email%");
            })
            ->when($phone, function ($q, $phone) {
                $q->where('phone', 'LIKE', "%$phone%");
            })
            ->when($roleId, function ($q, $roleId) {

                $q->where('role_id', $roleId);
            })
            ->when(isset($isActive) && $isActive !== '', function ($q) use ($isActive) {
                $q->where('is_active', (bool) $isActive);
            });

        $query->when($groupId, function ($q, $groupId) {

            $q->whereHas('groups', function ($subQuery) use ($groupId) {
                $subQuery->where('groups.id', $groupId);
            });
        });

        $query->when($serviceId, function ($q, $serviceId) {
            $q->whereHas('services', function ($subQuery) use ($serviceId) {
                $subQuery->where('services.id', $serviceId);
            });
        });

        $query->when($accommodationId, function ($q, $accommodationId) {

            $q->whereHas('accommodations', function ($subQuery) use ($accommodationId) {
                $subQuery->where('ac_accommodations.id', $accommodationId); // تأكد من اسم الجدول والـ ID
            });
        });

        $query->when($companyId, function ($q, $companyId) {
            $q->whereHas('companies', function ($subQuery) use ($companyId) {
                $subQuery->where('companies.id', $companyId);
            });
        });

        $query->when($currencyId, function ($q, $currencyId) {
            $q->whereHas('account_mappings', function ($subQuery) use ($currencyId) {
                $subQuery->where('currency_id', $currencyId);
            });
        });

        if ($column && array_key_exists($column, $this->columns)) {
            $query->orderBy($this->columns[$column], $order);
        } else {
            $query->orderBy('created_at', $order);
        }

        // 8. إرجاع النتيجة
        return $query->paginate($limit);
    }

    public function create(array $data)
    {
        // استخدام Transaction لضمان حفظ كل البيانات أو لا شيء
        return DB::transaction(function () use ($data) {

            // 1. إنشاء اليوزر الأساسي
            $user = User::create([
                'name' => $data['name'],
                'first_name' => $data['first_name'],
                'last_name' => $data['last_name'],
                'phone' => $data['phone'],
                'email' => $data['email'],
                'password' => $data['password'], // الـ Hash بيحصل تلقائي من $casts في الموديل
                'is_active' => $data['is_active'] ?? true,
                'employee_number' => $data['employee_number'] ?? null,
                'role_id' => $data['role_id'],
                'is_supplier' => $data['is_supplier'] ?? false,

            ]);

            // 2. ربط العلاقات (ManyToMany)
            // استخدام 'sync' آمن ومضمون
            if (! empty($data['groups'])) {
                $user->groups()->sync($data['groups']);
            }
            if (! empty($data['services'])) {
                $user->syncWithSourceColumn('services', $data['services']);
            }

            if (! empty($data['accommodations'])) {
                $user->syncWithSourceColumn('accommodations', $data['accommodations']);
            }
            if (! empty($data['companies'])) {
                $user->companies()->sync($data['companies']);
            }

            if (! empty($data['accounting'])) {

                $user->account_mappings()->createMany($data['accounting']);
            }

            return $user;
        });
    }

    public function find($id)
    {

        return User::with([
            'role',
            'groups',
            'services',
            'accommodations',
            'companies',
            'account_mappings.currency',
            'account_mappings.commissionAccount',
            'account_mappings.walletAccount',
            'account_mappings.supplierAccount',
        ])->find($id);
    }

    public function update($id, $data)
    {

        return DB::transaction(function () use ($id, $data) {

            $user = User::findOrFail($id);

            if (empty($data['password'])) {

                unset($data['password']);

            }

            $userData = [

                'name' => $data['name'] ?? null,

                'first_name' => $data['first_name'] ?? null,

                'last_name' => $data['last_name'] ?? null,

                'phone' => $data['phone'] ?? null,

                'email' => $data['email'] ?? null,

                'is_active' => $data['is_active'] ?? null,

                'employee_number' => $data['employee_number'] ?? null,

                'role_id' => $data['role_id'] ?? null,

                'is_supplier' => $data['is_supplier'] ?? null,

            ];

            $userData = array_filter($userData, fn ($value) => ! is_null($value));

            if (! empty($data['password'])) {

                $userData['password'] = bcrypt($data['password']);

            }

            $user->update($userData);

            if (isset($data['groups'])) {

                $user->groups()->sync($data['groups']);

            }

            if (isset($data['services'])) {
                $user->syncWithSourceColumn('services', $data['services']);
            }

            if (isset($data['accommodations'])) {
                $user->syncWithSourceColumn('accommodations', $data['accommodations']);
            }

            if (isset($data['companies'])) {

                $user->companies()->sync($data['companies']);

            }

            if (isset($data['accounting'])) {

                $user->account_mappings()->delete();

                foreach ($data['accounting'] as $mapping) {

                    $mapping['user_id'] = $user->id;

                    UserAccountMapping::create($mapping);

                }

            }

            return $user->refresh();

        });

    }

    public function delete($id)
    {
        $user = User::find($id);

        if ($user) {

            return $user->delete();
        }

        return false;
    }

    public function syncOverrides(int $id, array $overrides)
    {

        $user = $this->find($id);
        if (! $user) {
            return false;
        }

        $permissions = DB::table('permissions')
            ->whereIn('id', array_keys($overrides))
            ->select('id', 'Controller_name', 'collection_name')
            ->get();

        $syncData = [];

        foreach ($permissions as $permission) {
            $syncData[$permission->id] = [
                'override_type' => $overrides[$permission->id]['override_type'],
                'function_name' => $permission->Controller_name,
                'collection_name' => $permission->collection_name, ];
        }

        $user->overrides()->sync($syncData);

        return true;
    }

    public function setRole(int $id, ?int $roleId): bool
    {
        $user = $this->find($id);
        if ($user) {
            $user->role_id = $roleId;

            return $user->save();
        }

        return false;
    }
}
