<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CategoryTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class CategoryTranslationService extends Services
{
    private $categoryTranslation;

    public function __construct()
    {
        $this->categoryTranslation = new CategoryTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['category_id' => 'required|integer|exists:categories,id'];
        $combinations = [];
        $customErrors = [];
        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:category_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('category_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('category_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:300';
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Category.';
                } else {
                    $combinations[] = $key;
                }
            }
        }
        $validator = Validator::make(array_merge($data, ['category_id' => $id]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->categoryTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->categoryTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:category_translation,id',
        ];
        $validator = \Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->categoryTranslation->del($id);
    }
}
