<?php

namespace App\sys\Services\General\Translation;

use App\Models\General\LanguageTranslation;
use App\Models\Languages\Languages;
use App\sys\Repository\General\Translation\TranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;

class TranslationService extends Services
{
    protected $translationRepository;

    public function __construct(TranslationRepository $translationRepository)
    {
        $this->translationRepository = $translationRepository;
    }

    public function publish($languageId)
    {
        $language = Languages::find($languageId);

        if (! $language) {
            return false;
        }

        $translations = LanguageTranslation::where('language_id', $languageId)
            ->whereNotNull('value')
            ->get()
            ->groupBy('controller_name')
            ->map(function ($items) {
                return $items->pluck('value', 'key');
            })
            ->toArray();

        $directory = public_path('lang');
        if (! File::isDirectory($directory)) {
            File::makeDirectory($directory, 0755, true);
        }

        $filePath = public_path("lang/{$language->id}.json");

        File::put(
            $filePath,
            json_encode($translations, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
        );

        return true;
    }

    public function getPaginated($translation_id)
    {
        return $this->translationRepository->getPaginated($translation_id);
    }

    public function getControllerNames()
    {
        return $this->translationRepository->getUniqueControllerNames();
    }

    public function updateBatch($data)
    {
        $rules = [
            '*.language_id' => 'required|exists:languages,id',
            '*.translation_id' => 'required|exists:language_translations,id',
            '*.value' => 'required|string',
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $updatedCount = $this->translationRepository->updateBatch($data);

        if ($updatedCount > 0 && ! empty($data)) {
            $languageId = $data[0]['language_id'];
            $this->publish($languageId);
        }

        return $updatedCount;
    }

    public function store($data)
    {
        $rules = [
            'key' => 'required|string',
            'controller_name' => 'required|string',
            'value' => 'nullable|string',
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $success = $this->translationRepository->addKey(
            $data['key'],
            $data['controller_name'],
            $data['value'] ?? ''
        );

        if ($success) {
            $languages = Languages::all();
            foreach ($languages as $language) {
                $this->publish($language->id);
            }
        }

        return $success;
    }
}
