<?php

namespace App\sys\Services\Invoice;

use App\Models\invoice\InvoiceServices;
use App\Models\Profile\Profile;
use App\Models\Profile\ProfileTraveler;
use App\Models\UserAccountMapping;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class InvoiceServicesService extends Services
{
    protected InvoiceServicesRepository $invoiceServicesRepository;

    public function __construct(InvoiceServicesRepository $invoiceServicesRepository)
    {
        $this->invoiceServicesRepository = $invoiceServicesRepository;
    }

    public function create(array $request)
    {

        // Validate service type
        if (! isset($request['travel_tourism_type'])) {
            $this->setError(['travel_tourism_type' => ['Service type is required']]);

            return false;
        }

        $serviceType = $request['travel_tourism_type'];

        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            $this->setError(['travel_tourism_type' => ['Unsupported service type']]);

            return false;
        }

        // Get validation rules for the service type
        $rules = InvoiceServicesTypeConfig::getValidationRules($serviceType);
        // Merge type-specific handler rules
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $rules = array_merge($rules, $handler->getCreateRules($request));
        }
        // Add common required fields
        $rules['travel_tourism_type'] = 'required|string|in:'.implode(',', InvoiceServicesTypeConfig::getSupportedTypes());

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // Delegate to type handler to prepare data
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $prepared = $handler->prepareForCreate($request);
            if ($prepared === false) {
                $this->setError(['unknown' => ['unknown error in type handler']]);

                return false;
            }
            if (is_array($prepared) && array_key_exists('errors', $prepared)) {
                $this->setError($prepared['errors']);

                return false;
            }
            if (is_array($prepared)) {
                $request = $prepared;
            }
        }

        // Inject sales and handling flags based on profile
        if (isset($request['profile_id'])) {
            $profile = Profile::select(['id', 'user_id', 'has_invoice'])->find($request['profile_id']);
            if ($profile) {
                $request['salse_id'] = $profile->user_id;
                $request['is_by_handling'] = $profile->has_invoice ? 1 : 0;
            }
        }

        $created = $this->invoiceServicesRepository->create($request);
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $handler->afterCreate($created, $request);
        }

        return $created;
    }

    public function update(array $request)
    {
        // Require target id
        if (! array_key_exists('id', $request) || is_null($request['id'])) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail((int) $request['id']);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        // Get service type from existing item or request
        $serviceType = $request['travel_tourism_type'] ?? $item->travel_tourism_type;

        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            $this->setError(['travel_tourism_type' => ['Unsupported service type']]);

            return false;
        }

        // Get validation rules for the service type (only for provided fields)
        $rules = [];
        $providedFields = array_keys($request);

        foreach ($providedFields as $field) {
            $typeRules = InvoiceServicesTypeConfig::getValidationRules($serviceType);
            if (isset($typeRules[$field])) {
                $rules[$field] = $typeRules[$field];
            }
        }

        // Add common validation for provided fields
        if (isset($request['travel_tourism_type'])) {
            $rules['travel_tourism_type'] = 'required|string|in:'.implode(',', InvoiceServicesTypeConfig::getSupportedTypes());
        }
        if (isset($request['type'])) {
            $rules['type'] = 'required|string|in:credit,debit';
        }

        // Merge type-specific update rules
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $rules = array_merge($rules, $handler->getUpdateRules($request));
        }

        if (! empty($rules)) {
            $validator = Validator::make($request, $rules);

            if ($validator->fails()) {
                $this->setError($validator->errors());

                return false;
            }
        }

        // Delegate to type handler to prepare update data
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $prepared = $handler->prepareForUpdate($request, $item);
            if ($prepared === false) {
                $this->setError(['unknown' => ['unknown error in type handler']]);

                return false;
            }
            if (is_array($prepared) && array_key_exists('errors', $prepared)) {
                $this->setError($prepared['errors']);

                return false;
            }
            if (is_array($prepared)) {
                $request = $prepared;
            }
        }

        $updated = $this->invoiceServicesRepository->update($item, $request);
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $handler->afterUpdate($updated, $request);
        }

        return $updated;
    }

    public function delete(int $id)
    {
        if (is_null($id)) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail($id);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        return $this->invoiceServicesRepository->delete($item);
    }

    public function deleteMany(array $ids)
    {
        $validator = \Illuminate\Support\Facades\Validator::make(['ids' => $ids], [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:invoice_services,id'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceServicesRepository->deleteManyWithRules($ids);
    }

    public function getById(int $id)
    {
        if (is_null($id)) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail($id);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        return $item;
    }

    public function getByProfile(int $profileId, ?string $serviceType = null)
    {
        return $this->invoiceServicesRepository->getByProfile($profileId, $serviceType);
    }

    public function getByDaily(int $daily, ?string $serviceType = null)
    {
        return $this->invoiceServicesRepository->getByDaily($daily, $serviceType);
    }

    public function getList(array $filters = [])
    {
        return $this->invoiceServicesRepository->getList($filters);
    }

    public function getHandlingServices(array $filters = [])
    {
        return $this->invoiceServicesRepository->getHandlingServices($filters);
    }

    public function getByInvoice(int $invoiceId)
    {
        return $this->invoiceServicesRepository->getByInvoice($invoiceId);
    }

    public function getAllChanges(int $id)
    {
        $item = $this->getById($id);
        if (! $item) {
            return false;
        }

        return $this->invoiceServicesRepository->getAllChanges($item);
    }

    /**
     * Get validation rules for a specific service type
     */
    public function getValidationRulesForType(string $serviceType): array
    {
        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            return [];
        }

        return InvoiceServicesTypeConfig::getValidationRules($serviceType);
    }

    /**
     * Get supported service types
     */
    public function getSupportedServiceTypes(): array
    {
        return InvoiceServicesTypeConfig::getSupportedTypes();
    }

    /**
     * Ensure nationality exists under profile travelers and requested counts do not exceed stored counts
     */
    private function validateTravelerCounts(int $profileId, int $nationalityId, int $adultsRequested, int $childrenRequested)
    {
        $totals = ProfileTraveler::query()
            ->where('profile_id', $profileId)
            ->where('nationality_id', $nationalityId)
            ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
            ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
            ->first();

        if (! $totals || ((int) $totals->adults_total + (int) $totals->children_total) === 0) {
            return [
                'nationality_id' => ['لا يوجد مسافرون بهذه الجنسية مرتبطون بالبروفايل'],
            ];
        }

        $errors = [];
        if ($adultsRequested > (int) $totals->adults_total) {
            $errors['adults_count'][] = 'عدد الكبار المطلوب أكبر من المسجل في المسافرين';
        }
        if ($childrenRequested > (int) $totals->children_total) {
            $errors['children_count'][] = 'عدد الأطفال المطلوب أكبر من المسجل في المسافرين';
        }

        return empty($errors) ? true : $errors;
    }

    public function search()
    {
        return $this->invoiceServicesRepository->search();
    }

    /**
     * Search flights without linked profile_id using flexible filters.
     */
    public function searchFlightsWithoutProfile(array $filters)
    {
        $rules = [
            'pnr_number' => ['sometimes', 'string'],
            'office_id' => ['sometimes', 'integer'],
            'origin_airport' => ['sometimes', 'integer', 'exists:airports,id'],
            'origin_date' => ['sometimes', 'date'],
            'destination_airport' => ['sometimes', 'integer', 'exists:airports,id'],
            'destination_date' => ['sometimes', 'date'],
            'service_id' => ['sometimes', 'integer', 'exists:services,id'],
            'flight_company' => ['sometimes', 'integer', 'exists:airlines,id'],
            'limit' => ['sometimes', 'integer', 'min:1', 'max:200'],
        ];

        $validator = Validator::make($filters, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $validated = $validator->validated();
        $limit = $validated['limit'] ?? 20;
        unset($validated['limit']);

        return $this->invoiceServicesRepository->searchFlightsWithoutProfile($validated, $limit);
    }

    /**
     * Link a profile to multiple invoice services that currently have no profile_id.
     */
    public function assignProfileToServices(array $data)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'invoice_services_ids' => ['required', 'array', 'min:1'],
            'invoice_services_ids.*' => ['required', 'integer', 'distinct', 'exists:invoice_services,id'],
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $profile = Profile::select(['id', 'user_id', 'has_invoice'])->find($data['profile_id']);
        if (! $profile) {
            $this->setError(['profile_id' => ['Profile not found']]);

            return false;
        }

        $ids = array_values(array_unique($data['invoice_services_ids']));
        $services = InvoiceServices::whereIn('id', $ids)->get(['id', 'profile_id']);

        // Ensure all requested ids exist (in case soft deletes or other filters prevented retrieval)
        if ($services->count() !== count($ids)) {
            $missing = array_diff($ids, $services->pluck('id')->all());
            $this->setError(['invoice_services_ids' => ['Some ids not found: '.implode(',', $missing)]]);

            return false;
        }

        $alreadyLinked = $services->whereNotNull('profile_id')->pluck('id')->values()->all();
        if (! empty($alreadyLinked)) {
            $this->setError(['invoice_services_ids' => ['Services already linked to profiles: '.implode(',', $alreadyLinked)]]);

            return false;
        }

        return $this->invoiceServicesRepository->assignProfileToServices($ids, $profile);
    }

    public function paidSuppliers($data)
    {
        // Validation
        $rules = [
            'id' => ['required', 'integer', 'exists:invoice_services,id'],
            'amount' => ['required', 'numeric', 'min:0'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'accounting_id' => ['nullable', 'integer', 'exists:pay_type_accounting,id'],
            'pay_type_id' => ['required', 'integer', 'exists:pay_type,id'],
            'next_date_pay' => ['nullable', 'date'],
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // Get invoice service with supplier relationship
        $invoiceService = $this->invoiceServicesRepository->findByIdOrFail($data['id']);
        if (! $invoiceService) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }
        $invoiceService->load('supplier', 'company', 'profile');

        // Check supplier_id or guide_id based on service type
        if ($invoiceService->travel_tourism_type == 'tour_reps') {
            // For tour_reps: must have either supplier_id OR guide_id
            if (! $invoiceService->supplier_id && ! $invoiceService->guide_id) {
                $this->setError(['id' => ['This invoice service must have either a supplier or a guide']]);

                return false;
            }
        } else {
            // For other types: must have supplier_id
            if (! $invoiceService->supplier_id) {
                $this->setError(['id' => ['This invoice service does not have a supplier']]);

                return false;
            }
        }

        // Check currency_id matches
        if ($invoiceService->currency_id != $data['currency_id']) {
            $this->setError(['currency_id' => ['Currency does not match invoice service currency']]);

            return false;
        }

        // Calculate total amount
        $totalAmount = $invoiceService->travel_tourism_type === 'accommodation'
            ? (float) ($invoiceService->purchase_price ?? 0)
            : (float) ($invoiceService->grand_total ?? 0);

        if ($totalAmount <= 0) {
            $this->setError(['amount' => ['Service total amount is zero or invalid']]);

            return false;
        }

        // Get current paid amount
        $currentPaid = (float) ($invoiceService->paid_to_supplier ?? 0);
        $remainingAmount = $totalAmount - $currentPaid;

        // Validate amount
        if ($data['amount'] > $remainingAmount) {
            $this->setError(['amount' => ['Amount exceeds remaining amount. Remaining: '.$remainingAmount]]);

            return false;
        }

        // Check if already paid
        if ($invoiceService->is_paid == 1) {
            $this->setError(['id' => ['This service is already fully paid']]);

            return false;
        }

        // Get pay type
        $payType = \App\Models\Accounting\Paytype::find($data['pay_type_id']);
        if (! $payType) {
            $this->setError(['pay_type_id' => ['Pay type not found']]);

            return false;
        }

        // Determine tree_account_id based on pay_type
        $treeAccountId = null; // حساب الشركة/البنك (دائن)
        $debitAccountId = null; // حساب المورد (مدين)

        // Get supplier account (debit - مدين) - always needed
        if ($invoiceService->travel_tourism_type == 'tour_reps' && ! $invoiceService->guide_id) {
            $supplierMapping = UserAccountMapping::where('user_id', $invoiceService->guide_id)
                ->where('currency_id', $data['currency_id'])->first();
            if (! $supplierMapping && ! $supplierMapping->Supplier_account_id) {
                $this->setError(['emplayee' => ['No supplier account found for this currency']]);

                return false;
            }
            $debitAccountId = $supplierMapping->Supplier_account_id;
        } else {
            $supplierMapping = \App\Models\Suppliers\SupplierAccountMappings::where('supplier_id', $invoiceService->supplier_id)
                ->where('currency_id', $data['currency_id'])
                ->where('type', \App\sys\Enums\SupplierAccountMappings::START_BALANCE->value)
                ->first();
            if (! $supplierMapping || ! $supplierMapping->tree_account_id) {
                $this->setError(['supplier' => ['No supplier account found for this currency']]);

                return false;
            }
            $debitAccountId = $supplierMapping->tree_account_id;
        }

        // Get creditor account (دائن) based on pay_type if not provided

        switch ($payType->type) {
            case \App\sys\Enums\PaymentTypes::COMPANY_CASH->value:
                // Get from company_account_mappings with type = cash
                $companyId = $invoiceService->company_id ?? $invoiceService->profile->company_id ?? null;
                if (! $companyId) {
                    $this->setError(['company' => ['Company ID not found for this invoice service']]);

                    return false;
                }
                $companyMapping = \App\Models\General\CompanyAccountMappings::where('company_id', $companyId)
                    ->where('currency_id', $data['currency_id'])
                    ->where('type', \App\sys\Enums\CompanyAccountMapping::CACH->value)
                    ->first();
                if (! $companyMapping) {
                    $this->setError(['tree_account_id' => ['No cash account found for this company and currency']]);

                    return false;
                }
                $treeAccountId = $companyMapping->tree_account_id;
                break;

            case \App\sys\Enums\PaymentTypes::EMPLOYEE_CASH->value:
                // Get from user_account_mappings wallet_account_id
                $userMapping = \App\Models\UserAccountMapping::where('user_id', $invoiceService->salse_id ?? auth()->id())
                    ->where('currency_id', $data['currency_id'])
                    ->first();
                if (! $userMapping || ! $userMapping->wallet_account_id) {
                    $this->setError(['tree_account_id' => ['No wallet account found for this user and currency']]);

                    return false;
                }
                $treeAccountId = $userMapping->wallet_account_id;
                break;

            case \App\sys\Enums\PaymentTypes::BANK->value:
                // Must provide tree_account_id
                $payTypeAccounting = \App\Models\Accounting\PaytypeAccounting::find($data['accounting_id']);
                if (! $payTypeAccounting) {
                    $this->setError(['accounting_id' => ['tree_account_id is required for bank payment type']]);

                    return false;
                }
                $treeAccountId = $payTypeAccounting->tree_accounting_transfer_id;
                break;

            case \App\sys\Enums\PaymentTypes::PAYMENT_METHOD_RATES->value:
                // Get from pay_type_accounting tree_accounting_transfer_id (this will be creditor)
                $payTypeAccounting = \App\Models\Accounting\PaytypeAccounting::find($data['accounting_id']);
                if (! $payTypeAccounting) {
                    $this->setError(['pay_type_id' => ['No accounting found for this pay type and currency']]);

                    return false;
                }
                $treeAccountId = $payTypeAccounting->tree_accounting_transfer_id;
                break;
        }

        // Get currency for rate calculation
        $currency = \App\Models\General\Currency::find($data['currency_id']);
        if (! $currency) {
            $this->setError(['currency_id' => ['Currency not found']]);

            return false;
        }

        $exchangeRate = (float) ($currency->is_default == 1 ? 1 : $currency->exchange_rate);

        // Calculate amounts
        $amountInDefaultCurrency = $data['amount'] * $exchangeRate;

        // Get supplier name
        $supplierName = $invoiceService->supplier && $invoiceService->supplier->supplier_name
            ? $invoiceService->supplier->supplier_name
            : 'Supplier';

        // Create transfers for constraint
        $transfers = [
            [
                'tree_accounting_id' => $debitAccountId, // حساب المورد (مدين)
                'currency_id' => $data['currency_id'],
                'name' => 'دفع مورد - '.$supplierName,
                'debit' => round($amountInDefaultCurrency, 2),
                'creditor' => 0,
                'currency_debit' => round($data['amount'], 2),
                'currency_creditor' => 0,
                'cost_center_id' => null,
            ],
            [
                'tree_accounting_id' => $treeAccountId, // حساب الشركة/البنك (دائن)
                'currency_id' => $data['currency_id'],
                'name' => 'دفع مورد - '.$supplierName,
                'debit' => 0,
                'creditor' => round($amountInDefaultCurrency, 2),
                'currency_debit' => 0,
                'currency_creditor' => round($data['amount'], 2),
                'cost_center_id' => null,
            ],
        ];

        $totalDebit = array_sum(array_column($transfers, 'debit'));
        $totalCreditor = array_sum(array_column($transfers, 'creditor'));

        // Create constraint
        $constraintData = [
            'supplier_id' => $invoiceService->supplier_id,
            'profile_id' => $invoiceService->profile_id,
            'date' => date('Y-m-d'),
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'name' => 'دفع مورد',
            'description' => 'دفع مورد - خدمة رقم '.$invoiceService->id,
            'active' => 1,
            'capture_exchange' => 'payment',
            'creation_mode' => 'automatic',
            'currency_id' => $data['currency_id'],
            'currency_transfer_rate' => $exchangeRate,
            'company_id' => $invoiceService->company_id ?? $invoiceService->profile->company_id ?? null,
            'transfers' => $transfers,
        ];

        $constraintRepo = new \App\sys\Repository\Accounting\ConstraintRepository;
        $constraintRepo->create($constraintData, 'yes');

        // Update invoice service
        $newPaidAmount = $currentPaid + $data['amount'];
        $newRemainingAmount = $totalAmount - $newPaidAmount;
        $isPaid = $newRemainingAmount <= 0.01 ? 1 : 0; // Consider paid if remaining is less than 0.01

        $updateData = [
            'paid_to_supplier' => round($newPaidAmount, 2),
            'remaining_to_supplier' => round($newRemainingAmount, 2),
            'is_paid' => $isPaid,
            'next_pay_date' => $data['next_date_pay'],
        ];

        $this->invoiceServicesRepository->update($invoiceService, $updateData);

        return true;
    }

    public function GetSummary($profileId)
    {
        return $this->invoiceServicesRepository->GetSummary($profileId);

    }

    /**
     * Update service status (pending, confirmed, cancelled, done) and ensure accounting constraint logic
     * Same logic as AccommodationReservationService->updateStatus but with InvoiceServices specific rules:
     * - Reject if travel_tourism_type == accommodation
     * - For tour_reps: supplier can be supplier_id OR guide_id
     * - For other types: supplier_id is required
     * - If status == confirmed, create purchase invoice constraint
     */
    public function updateStatus(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:invoice_services,id'],
            'status' => ['required', 'in:pending,confirmed,cancelled,done'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $invoiceService = $this->invoiceServicesRepository->findByIdOrFail((int) $request['id']);
        if (! $invoiceService) {
            $this->setError(['id' => ['الخدمة غير موجودة']]);

            return false;
        }

        // Reject accommodation services
        if ($invoiceService->travel_tourism_type === 'accommodation') {
            $this->setError(['status' => ['لا يمكن تغيير حالة خدمات الإقامة من هنا']]);

            return false;
        }

        $currentStatus = $invoiceService->status ?? 'pending';
        $newStatus = $request['status'];

        // التحقق من قواعد تغيير الحالة
        if (! $this->canChangeStatus($currentStatus, $newStatus)) {
            $this->setError(['status' => ['لا يمكن تغيير الحالة من '.$currentStatus.' إلى '.$newStatus]]);

            return false;
        }

        // إذا كان التحويل إلى confirmed، التحقق من وجود supplier_id/guide_id وإنشاء القيد
        if ($newStatus === 'confirmed') {
            // For tour_reps: check guide_id or supplier_id
            if ($invoiceService->travel_tourism_type === 'tour_reps') {
                if (! $invoiceService->supplier_id && ! $invoiceService->guide_id) {
                    $this->setError(['status' => ['يجب تحديد مورد أو مرشد (guide) لهذه الخدمة قبل التأكيد']]);

                    return false;
                }
            } else {
                // For other types: supplier_id is required
                if (! $invoiceService->supplier_id) {
                    $this->setError(['status' => ['يجب تحديد المورد قبل تأكيد الخدمة']]);

                    return false;
                }
            }

            // إنشاء القيد المحاسبي
            if (! $this->createPurchaseInvoiceConstraint($invoiceService)) {
                return false;
            }
        }

        // تحديث الحالة (منفصلة عن update العادي)
        $updated = $this->invoiceServicesRepository->updateStatus($invoiceService, $newStatus);

        return $updated;
    }

    /**
     * Check if status change is allowed (same logic as AccommodationReservationService)
     */
    private function canChangeStatus(string $currentStatus, string $newStatus): bool
    {
        // done و cancelled لا يمكن تغييرهم
        if (in_array($currentStatus, ['done', 'cancelled'])) {
            return false;
        }

        // pending يمكن التحويل إلى confirmed أو cancelled فقط
        if ($currentStatus === 'pending') {
            return in_array($newStatus, ['confirmed', 'cancelled']);
        }

        // confirmed يمكن التحويل إلى cancelled أو done فقط (لا يمكن الرجوع لـ pending)
        if ($currentStatus === 'confirmed') {
            return in_array($newStatus, ['cancelled', 'done']);
        }

        return false;
    }

    /**
     * Create purchase invoice constraint (delegate to repository)
     */
    private function createPurchaseInvoiceConstraint($invoiceService): bool
    {
        return $this->invoiceServicesRepository->createPurchaseInvoiceConstraint($invoiceService);
    }
}
