<?php

namespace App\sys\Services\Invoice;

use App\sys\Enums\ServicesType;
use App\sys\Services\Invoice\Types\AdditionalServicesHandler;
use App\sys\Services\Invoice\Types\AttractionsTemplesHandler;
use App\sys\Services\Invoice\Types\DailyTransportationHandler;
use App\sys\Services\Invoice\Types\DiningEntertainmentHandler;
use App\sys\Services\Invoice\Types\FlightsHandler;
use App\sys\Services\Invoice\Types\InvoiceTypeHandlerInterface;
use App\sys\Services\Invoice\Types\SingleTransportationHandler;
use App\sys\Services\Invoice\Types\TourGuideHandler;
use App\sys\Services\Invoice\Types\TourRepsHandler;

class InvoiceServicesTypeConfig
{
    /**
     * Get configuration for each service type
     */
    public static function getTypeConfig(): array
    {
        return [
            ServicesType::ATTRACTIONS_TEMPLES->value => [
                'required_fields' => [
                    'daily_program_id' => 'required|integer|exists:pr_daily_programs,id',
                    'nationality_id' => 'nullable|integer|exists:nationalities,id',
                    'adults_count' => 'required|integer|min:0',
                    'children_count' => 'required|integer|min:0',
                    // totals are computed internally in handler
                    'executive_id' => 'nullable|integer|exists:users,id',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    // 'currency_rate' => 'required|numeric|min:0',
                    'is_by_handling' => 'nullable|boolean',              ],
                'optional_fields' => [
                    'service_id' => 'nullable|integer|exists:services,id',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'invoice_id' => 'nullable|integer',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'adult_price' => 'nullable|numeric|min:0',
                    'child_price' => 'nullable|numeric|min:0',
                    'reserve_number' => 'nullable|string|max:255',

                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'daily_program_id',
                    'nationality_id',
                    'adults_count',
                    'children_count',
                    'adult_price',
                    'child_price',
                    'adult_total',
                    'child_total',
                    'grand_total',
                    'executive_id',
                    'currency_id',
                    'currency_rate',
                    'service_id',
                    'profile_id',
                    'invoice_id',
                    'supplier_id',
                    'reserve_number',
                    'notes',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'dailyProgram',
                    'nationality',
                    'executive',
                    'currency',
                    'service',
                    'profile',
                    'supplier',
                ],
            ],

            ServicesType::DINING_ENTERTAINMENT->value => [
                'required_fields' => [
                    'daily_program_id' => 'required|integer|exists:pr_daily_programs,id',
                    'service_id' => 'required|integer|exists:services,id',
                    'nationality_id' => 'nullable|integer|exists:nationalities,id',
                    'adults_count' => 'required|integer|min:0',
                    'children_count' => 'required|integer|min:0',
                    'adult_price' => 'required|numeric|min:0',
                    'child_price' => 'required|numeric|min:0',
                    'is_by_handling' => 'nullable|boolean',

                    'executive_id' => 'nullable|integer|exists:users,id',
                    // 'currency_rate' => 'required|numeric|min:0',
                ],
                'optional_fields' => [
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'invoice_id' => 'nullable|integer',
                    'adult_total' => 'nullable|numeric|min:0',
                    'child_total' => 'nullable|numeric|min:0',
                    'reserve_number' => 'nullable|string|max:255',
                    'notes' => 'nullable|string|max:1000',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'daily_program_id',
                    'service_id',
                    'supplier_id',
                    'nationality_id',
                    'adults_count',
                    'children_count',
                    'adult_price',
                    'child_price',
                    'adult_total',
                    'child_total',
                    'grand_total',
                    'executive_id',
                    'currency_id',
                    'currency_rate',
                    'profile_id',
                    'invoice_id',
                    'reserve_number',
                    'notes',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'dailyProgram',
                    'service',
                    'supplier',
                    'nationality',
                    'executive',
                    'currency',
                    'profile',
                ],
            ],

            ServicesType::TOUR_GUIDE->value => [
                'required_fields' => [
                    'service_id' => 'required|integer|exists:services,id',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'category_id' => 'required|integer|exists:categories,id',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'daily_program_id' => 'nullable|integer|exists:pr_daily_programs,id',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'people_count' => 'nullable|integer|min:0',
                    'daily_cost' => 'nullable|numeric|min:0',
                    'tip_amount' => 'nullable|numeric|min:0',
                    'total_tips' => 'nullable|numeric|min:0',
                    'total_tax' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'notes' => 'nullable|string|max:1000',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'service_id',
                    'profile_id',
                    'currency_id',
                    'currency_rate',
                    'daily_program_id',
                    'supplier_id',
                    'category_id',
                    'people_count',
                    'daily_cost',
                    'tip_amount',
                    'total_tips',
                    'total_tax',
                    'grand_total',
                    'notes',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'service',
                    'profile',
                    'currency',
                    'dailyProgram',
                    'taxes',
                ],
            ],

            ServicesType::ACCOMMODATION->value => [
                'required_fields' => [
                    'service_id' => 'required|integer|exists:services,id',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'currency_id' => 'required|integer|exists:currencies,id',
                    'currency_rate' => 'required|numeric|min:0',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'daily_program_id' => 'nullable|integer|exists:pr_daily_programs,id',
                    'nationality_id' => 'nullable|integer|exists:nationalities,id',
                    'adults_count' => 'nullable|integer|min:0',
                    'children_count' => 'nullable|integer|min:0',
                    'adult_price' => 'nullable|numeric|min:0',
                    'child_price' => 'nullable|numeric|min:0',
                    'adult_total' => 'nullable|numeric|min:0',
                    'child_total' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'executive_id' => 'nullable|integer|exists:users,id',
                    'invoice_id' => 'nullable|integer',
                    'supplier_id' => 'nullable|integer',
                    'reserve_number' => 'nullable|string|max:255',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'service_id',
                    'profile_id',
                    'currency_id',
                    'currency_rate',
                    'daily_program_id',
                    'nationality_id',
                    'adults_count',
                    'children_count',
                    'adult_price',
                    'child_price',
                    'adult_total',
                    'child_total',
                    'grand_total',
                    'executive_id',
                    'invoice_id',
                    'supplier_id',
                    'reserve_number',
                    'created',
                    'modified',
                    'is_by_handling',
                ],
                'relationships' => [
                    'service',
                    'profile',
                    'currency',
                    'dailyProgram',
                    'nationality',
                    'executive',
                ],
            ],

            ServicesType::DAILY_TRANSPORTATION->value => [
                'required_fields' => [
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'daily_program_id' => 'required|integer|exists:pr_daily_programs,id',
                    'city_id' => 'required|integer|exists:cities,id',
                    'service_id' => 'required|integer|exists:services,id',
                    'people_count' => 'required|integer|min:0',
                    // total_tax is computed internally by handler
                    'operation_data_id' => 'required|integer|exists:ge_operation_data,id',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'daily_cost' => 'nullable|numeric|min:0',
                    'extra_cost' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'currency_rate' => 'nullable|numeric|min:0',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'executive_id' => 'nullable|integer|exists:users,id',
                    'notes' => 'nullable|string|max:1000',
                    'invoice_id' => 'nullable|integer',
                    'reserve_number' => 'nullable|string|max:255',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'profile_id',
                    'daily_program_id',
                    'supplier_id',
                    'city_id',
                    'service_id',
                    'people_count',
                    'daily_cost',
                    'extra_cost',
                    'total_tax',
                    'grand_total',
                    'currency_id',
                    'currency_rate',
                    'operation_data_id',
                    'executive_id',
                    'notes',
                    'invoice_id',
                    'reserve_number',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'profile',
                    'dailyProgram',
                    'supplier',
                    'city',
                    'service',
                    'currency',
                    'operationData',
                    'taxes',
                ],
            ],

            ServicesType::SINGLE_TRANSPORTATION->value => [
                'required_fields' => [
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'city_id' => 'required|integer|exists:cities,id',
                    'city_to_id' => 'required|integer|exists:cities,id',
                    'service_id' => 'required|integer|exists:services,id',
                    'people_count' => 'required|integer|min:0',
                    'operation_data_id' => 'required|integer|exists:ge_operation_data,id',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'daily_cost' => 'nullable|numeric|min:0',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'extra_cost' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'currency_rate' => 'nullable|numeric|min:0',
                    'executive_id' => 'nullable|integer|exists:users,id',
                    'notes' => 'nullable|string|max:1000',
                    'invoice_id' => 'nullable|integer',
                    'reserve_number' => 'nullable|string|max:255',
                    'execution_date' => 'nullable|date',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'profile_id',
                    'supplier_id',
                    'city_id',
                    'city_to_id',
                    'service_id',
                    'people_count',
                    'daily_cost',
                    'extra_cost',
                    'total_tax',
                    'grand_total',
                    'currency_id',
                    'currency_rate',
                    'operation_data_id',
                    'executive_id',
                    'notes',
                    'invoice_id',
                    'reserve_number',
                    'created',
                    'modified',
                    'is_by_handling',
                    'execution_date',

                ],
                'relationships' => [
                    'profile',
                    'supplier',
                    'city',
                    'cityTo',
                    'service',
                    'currency',
                    'operationData',
                    'taxes',
                ],
            ],

            ServicesType::TOUR_REPS->value => [
                'required_fields' => [
                    'service_id' => 'required|integer|exists:services,id',
                    'people_count' => 'required|integer|min:0',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'daily_program_id' => 'required|integer|exists:pr_daily_programs,id',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'daily_program_id' => 'nullable|integer|exists:pr_daily_programs,id',
                    'guide_language_id' => 'nullable|integer|exists:languages,id',
                    'service_provider' => 'nullable|in:supplier_id,guide_id',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'guide_id' => 'nullable|integer|exists:users,id',
                    'daily_cost' => 'nullable|numeric|min:0',
                    'tip_amount' => 'nullable|numeric|min:0',
                    'total_tips' => 'nullable|numeric|min:0',
                    'extra_cost' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'executive_id' => 'nullable|integer|exists:users,id',
                    'notes' => 'nullable|string|max:1000',
                    'invoice_id' => 'nullable|integer',
                    'reserve_number' => 'nullable|string|max:255',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'service_id',
                    'supplier_id',
                    'service_provider',
                    'guide_id',
                    'guide_language_id',
                    'people_count',
                    'daily_cost',
                    'tip_amount',
                    'total_tips',
                    'extra_cost',
                    'grand_total',
                    'executive_id',
                    'currency_id',
                    'currency_rate',
                    'profile_id',
                    'daily_program_id',
                    'invoice_id',
                    'reserve_number',
                    'notes',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'service',
                    'supplier',
                    'currency',
                    'profile',
                    'dailyProgram',
                    'guideLanguage',
                    'guide',
                ],
            ],

            ServicesType::ADDITIONAL_SERVICES->value => [
                'required_fields' => [
                    'service_id' => 'required|integer|exists:services,id',
                    'profile_id' => 'required|integer|exists:pr_profile,id',
                    'people_count' => 'required|integer|min:0',
                    'execution_date' => 'required|date',
                    'is_by_handling' => 'nullable|boolean',

                ],
                'optional_fields' => [
                    'nationality_id' => 'nullable|integer|exists:nationalities,id',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'currency_rate' => 'nullable|numeric|min:0',
                    'daily_cost' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'notes' => 'nullable|string|max:1000',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'executive_id' => 'nullable|integer|exists:users,id',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'currency_id',
                    'currency_rate',
                    'service_id',
                    'profile_id',
                    'nationality_id',
                    'people_count',
                    'daily_cost',
                    'total_tax',
                    'grand_total',
                    'execution_date',
                    'notes',
                    'supplier_id',
                    'executive_id',
                    'created',
                    'modified',
                    'is_by_handling',

                ],
                'relationships' => [
                    'service',
                    'profile',
                    'currency',
                    'nationality',
                    'supplier',
                    'executive',
                    'taxes',
                ],
            ],

            ServicesType::FLIGHTS->value => [
                'required_fields' => [
                    'service_id' => 'required|integer|exists:services,id',
                    'pnr_number' => 'required|string|max:255',
                    'ticket_date' => 'required|date',
                    'direction_type' => 'required|string|in:one_way,round_trip',
                    'is_by_handling' => 'nullable|boolean',
                ],
                'optional_fields' => [
                    'profile_id' => 'sometimes|nullable|integer|exists:pr_profile,id',
                    'domain_id' => 'nullable|integer',
                    'office_id' => 'nullable|string|max:50',
                    // economy_class الآن يقبل جميع درجات الحجز (First/Business/Premium/Economy booking classes)
                    'economy_class' => 'nullable|string|in:F,P,A,R,J,C,D,Z,I,W,E,Y,B,H,K,M,L,V,S,N,Q,O,T,X,G',
                    'currency_id' => 'nullable|integer|exists:currencies,id',
                    'currency_rate' => 'nullable|numeric|min:0',
                    'grand_total' => 'nullable|numeric|min:0',
                    'total_tax' => 'nullable|numeric|min:0',
                    'supplier_amount' => 'nullable|numeric|min:0',
                    'notes' => 'nullable|string|max:1000',
                    'supplier_id' => 'nullable|integer|exists:su_supplier,id',
                    'executive_id' => 'nullable|integer|exists:users,id',
                ],
                'resource_fields' => [
                    'id',
                    'travel_tourism_type',
                    'service_id',
                    'profile_id',
                    'pnr_number',
                    'office_id',
                    'ticket_date',
                    'domain_id',
                    'direction_type',
                    'economy_class',
                    'currency_id',
                    'currency_rate',
                    'total_tax',
                    'grand_total',
                    'supplier_amount',
                    'notes',
                    'supplier_id',
                    'executive_id',
                    'created',
                    'modified',
                    'is_by_handling',
                ],
                'relationships' => [
                    'service',
                    'profile',
                    'currency',
                    'supplier',
                    'executive',
                    'taxes',
                    'segments',
                    'airlineTickets',
                ],
            ],
        ];
    }

    /**
     * Get validation rules for a specific service type
     */
    public static function getValidationRules(string $type): array
    {
        $config = self::getTypeConfig();

        if (! isset($config[$type])) {
            return [];
        }

        return array_merge(
            $config[$type]['required_fields'],
            $config[$type]['optional_fields']
        );
    }

    /**
     * Get required fields for a specific service type
     */
    public static function getRequiredFields(string $type): array
    {
        $config = self::getTypeConfig();

        if (! isset($config[$type])) {
            return [];
        }

        return array_keys($config[$type]['required_fields']);
    }

    /**
     * Get resource fields for a specific service type
     */
    public static function getResourceFields(string $type): array
    {
        $config = self::getTypeConfig();

        if (! isset($config[$type])) {
            return [];
        }

        // Always include status in resource output
        return array_merge($config[$type]['resource_fields'], ['status']);
    }

    /**
     * Get relationships for a specific service type
     */
    public static function getRelationships(string $type): array
    {
        $config = self::getTypeConfig();

        if (! isset($config[$type])) {
            return [];
        }

        return $config[$type]['relationships'];
    }

    /**
     * Check if a service type is supported
     */
    public static function isSupportedType(string $type): bool
    {
        return array_key_exists($type, self::getTypeConfig());
    }

    /**
     * Get all supported service types
     */
    public static function getSupportedTypes(): array
    {
        return array_keys(self::getTypeConfig());
    }

    /**
     * Resolve type-specific handler (strategy)
     */
    public static function resolveHandler(string $type): ?InvoiceTypeHandlerInterface
    {
        return match ($type) {
            ServicesType::ATTRACTIONS_TEMPLES->value => new AttractionsTemplesHandler,
            ServicesType::DINING_ENTERTAINMENT->value => new DiningEntertainmentHandler,
            ServicesType::TOUR_GUIDE->value => new TourGuideHandler,
            ServicesType::DAILY_TRANSPORTATION->value => new DailyTransportationHandler,
            ServicesType::SINGLE_TRANSPORTATION->value => new SingleTransportationHandler,
            ServicesType::ADDITIONAL_SERVICES->value => new AdditionalServicesHandler,
            ServicesType::TOUR_REPS->value => new TourRepsHandler,
            ServicesType::FLIGHTS->value => new FlightsHandler,
            default => null,
        };
    }
}
