<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\General\TaxRate;
use App\sys\Enums\ServicesType;
use Illuminate\Support\Facades\Gate;

class AdditionalServicesHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        // Authorize basic fields
        $basic = $this->authorizeBasic($request, 'add');

        // Authorize financial fields if provided
        $finacial = $this->authorizeFinancial($request, 'add');

        return [
            'travel_tourism_type' => ['required', 'in:'.ServicesType::ADDITIONAL_SERVICES->value],
            'currency_id' => [$finacial, 'integer', 'exists:currencies,id'],
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'nationality_id' => [$basic, 'integer', 'exists:nationalities,id'],
            'people_count' => ['required', 'integer', 'min:0'],
            'daily_cost' => [$finacial, 'numeric', 'min:0'],
            'execution_date' => ['required', 'date'],
            'supplier_id' => [$finacial, 'exists:su_supplier,id'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'is_by_handling' => ['nullable', 'boolean'],
            // taxes
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            // computed/prohibited
            'currency_rate' => 'prohibited',
            'grand_total' => 'prohibited',
            'total_tax' => 'prohibited',
        ];
    }

    public function getUpdateRules(array $request): array
    {
        // Authorize basic fields
        $basic = $this->authorizeBasic($request, 'edit');

        // Authorize financial fields if provided
        $finacial = $this->authorizeFinancial($request, 'edit');

        return [
            'currency_id' => ['sometimes', $finacial, 'integer', 'exists:currencies,id'],
            'service_id' => ['sometimes', 'required', 'integer', 'exists:services,id'],
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'nationality_id' => ['sometimes', $basic, 'exists:nationalities,id'],
            'people_count' => ['sometimes', $basic, 'integer', 'min:0'],
            'daily_cost' => ['sometimes', $finacial, 'numeric', 'min:0'],
            'execution_date' => ['sometimes', 'required', 'date'],
            'supplier_id' => ['sometimes', $finacial, 'integer', 'exists:su_supplier,id'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'is_by_handling' => ['nullable', 'boolean'],
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];
    }

    public function prepareForCreate(array $request): array|false
    {
        // Service type optional validation if needed
        $service = GeneralService::find($request['service_id']);
        if ($service && $service->type !== ServicesType::ADDITIONAL_SERVICES->value) {
            return ['errors' => ['service_id' => ['service type must be additional_services']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        if (Gate::allows('ADD_OTHER_SERVICE_FINANCIAL_INFO')) {
            // Currency rate - handle optional currency_id
            $currencyId = (int) ($request['currency_id'] ?? null);
            if ($currencyId) {
                $currency = Currency::find($currencyId);
                $request['currency_rate'] = $currency?->exchange_rate ?? 1;
            } else {
                $request['currency_rate'] = 1;
            }

            $people = (int) ($request['people_count'] ?? 0);
            $dailyCost = (float) ($request['daily_cost'] ?? 0);
            $subtotal = $people * $dailyCost;
            $totalTax = 0.0;
            $taxRateIds = $request['tax_rate_id'] ?? [];
            if (is_array($taxRateIds) && ! empty($taxRateIds)) {
                $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $totalTax = ($subtotal * $sumPercent) / 100.0;
            }
            $request['total_tax'] = $totalTax;
            $request['grand_total'] = $subtotal + $totalTax;
        } else {
            $request['total_tax'] = 0;
            $request['grand_total'] = 0;
            $request['currency_rate'] = 1;
        }

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if ($service && $service->type !== ServicesType::ADDITIONAL_SERVICES->value) {
                return ['errors' => ['service_id' => ['service type must be additional_services']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }
        if (Gate::allows('EDIT_OTHER_SERVICE_FINANCIAL_INFO')) {
            $people = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
            $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
            $subtotal = $people * $dailyCost;
            $taxRateIds = $request['tax_rate_id'] ?? null;
            if (is_array($taxRateIds)) {
                $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $totalTax = ($subtotal * $sumPercent) / 100.0;
            } else {
                $totalTax = (float) ($existing->total_tax ?? 0);
            }
            $request['total_tax'] = $totalTax;
            $request['grand_total'] = $subtotal + $totalTax;
        }

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        $this->saveTaxes($invoiceService, $request);
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        if (array_key_exists('tax_rate_id', $request)) {
            $this->saveTaxes($invoiceService, $request, true);
        }
    }

    private function saveTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (! is_array($taxRateIds)) {
            return;
        }

        if ($replace) {
            \App\Models\invoice\InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $people = (int) ($request['people_count'] ?? $invoiceService->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $invoiceService->daily_cost ?? 0);
        $subtotal = $people * $dailyCost;

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($subtotal * (float) $rate->percentage) / 100.0;
            \App\Models\invoice\InvoiceServicesTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'percentage' => (float) $rate->percentage,
                'tax_amount' => $amount,
            ]);
        }
    }

    private function authorizeBasic(array $request, string $mode)
    {

        $ability = $mode === 'edit' ? 'EDIT_OTHER_SERVICE_BASIC_INFO' : 'ADD_OTHER_SERVICE_BASIC_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
        $fields = array_intersect(array_keys($request), ['service_id', 'nationality_id', 'people_count', 'execution_date']);
        if (empty($fields)) {
            return;
        }

        if (Gate::denies($ability)) {
            throw_if(true, \Illuminate\Auth\Access\AuthorizationException::class, __('auth.unauthorized'));
        }
    }

    private function authorizeFinancial(array $request, string $mode)
    {
        $ability = $mode === 'edit' ? 'EDIT_OTHER_SERVICE_FINANCIAL_INFO' : 'ADD_OTHER_SERVICE_FINANCIAL_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
        $fields = array_intersect(array_keys($request), ['daily_cost', 'supplier_id', 'currency_id']);
        if (empty($fields)) {
            return;
        }

        if (Gate::denies($ability)) {
            throw_if(true, \Illuminate\Auth\Access\AuthorizationException::class, __('auth.unauthorized'));
        }
    }
}
