<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\Profile\ProfileTraveler;
use App\sys\Enums\ServicesType;
use Illuminate\Support\Facades\Gate;

class DiningEntertainmentHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'add');
        $financial = $this->authorizeFinancial($request, 'add');

        return [
            'service_id' => [$basic, 'integer', 'exists:services,id'],
            'supplier_id' => [$financial, 'integer', 'exists:su_supplier,id'],
            'nationality_id' => [$basic, 'integer', 'exists:nationalities,id'],
            'adults_count' => [$basic, 'integer', 'min:0'],
            'children_count' => [$basic, 'integer', 'min:0'],
            'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'adult_price' => [$financial, 'numeric', 'min:0'],
            'child_price' => [$financial, 'numeric', 'min:0'],
            'currency_rate' => 'prohibited',
            'adult_total' => ['sometimes', $financial, 'numeric', 'min:0'],
            'child_total' => ['sometimes', $financial, 'numeric', 'min:0'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'is_by_handling' => ['nullable', 'boolean'],

        ];
    }

    public function getUpdateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'edit');
        $financial = $this->authorizeFinancial($request, 'edit');

        $rules = [
            'adult_total' => ['sometimes', $financial, 'numeric', 'min:0'],
            'child_total' => ['sometimes', $financial, 'numeric', 'min:0'],
            'grand_total' => 'prohibited',
            'currency_rate' => 'prohibited',
            'is_by_handling' => ['nullable', 'boolean'],

        ];

        if (array_key_exists('service_id', $request)) {
            $rules['service_id'] = [$basic, 'integer', 'exists:services,id'];
        }
        if (array_key_exists('supplier_id', $request)) {
            $rules['supplier_id'] = [$financial, 'integer', 'exists:su_supplier,id'];
        }
        if (array_key_exists('nationality_id', $request)) {
            $rules['nationality_id'] = [$basic, 'integer', 'exists:nationalities,id'];
        }
        if (array_key_exists('adults_count', $request)) {
            $rules['adults_count'] = [$basic, 'integer', 'min:0'];
        }
        if (array_key_exists('children_count', $request)) {
            $rules['children_count'] = [$basic, 'integer', 'min:0'];
        }
        if (array_key_exists('adult_price', $request)) {
            $rules['adult_price'] = [$financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('child_price', $request)) {
            $rules['child_price'] = [$financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('currency_id', $request)) {
            $rules['currency_id'] = [$financial, 'integer', 'exists:currencies,id'];
        }
        if (array_key_exists('profile_id', $request)) {
            $rules['profile_id'] = ['required', 'integer', 'exists:pr_profile,id'];
        }
        if (array_key_exists('notes', $request)) {
            $rules['notes'] = ['nullable', 'string', 'max:1000'];
        }
        if (array_key_exists('is_by_handling', $request)) {
            $rules['is_by_handling'] = ['nullable', 'boolean'];
        }

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        $this->authorizeBasic($request, 'add');
        $this->authorizeFinancial($request, 'add');
        $canFinancial = Gate::allows('ADD_RESTURANT_ENTERTAINMENT_FINANCIAL_INFO');

        // Validate service type
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::DINING_ENTERTAINMENT->value) {
            return ['errors' => ['service_id' => ['service type must be dining_entertainment']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Validate traveler counts by nationality under profile
        $countsOk = $this->validateTravelerCounts(
            (int) $request['profile_id'],
            (int) $request['nationality_id'],
            (int) $request['adults_count'],
            (int) $request['children_count']
        );
        if ($countsOk !== true) {
            return ['errors' => $countsOk]; // wrap errors
        }

        if (! $canFinancial) {
            $request['currency_rate'] = 1;
            $request['adult_total'] = 0;
            $request['child_total'] = 0;
            $request['grand_total'] = 0;

            return $request;
        }

        // Set currency rate
        $currency = isset($request['currency_id']) ? Currency::find($request['currency_id']) : null;
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Compute totals
        $adultTotal = isset($request['adult_total'])
            ? (float) $request['adult_total']
            : ((float) ($request['adult_price'] ?? 0)) * (int) $request['adults_count'];
        $childTotal = isset($request['child_total'])
            ? (float) $request['child_total']
            : ((float) ($request['child_price'] ?? 0)) * (int) $request['children_count'];
        $request['adult_total'] = $adultTotal;
        $request['child_total'] = $childTotal;
        $request['grand_total'] = $adultTotal + $childTotal;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        $this->authorizeBasic($request, 'edit');
        $this->authorizeFinancial($request, 'edit');
        $canFinancial = Gate::allows('EDIT_RESTURANT_ENTERTAINMENT_FINANCIAL_INFO');

        // If service_id provided or different, ensure it's correct type
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::DINING_ENTERTAINMENT->value) {
                return ['errors' => ['service_id' => ['service type must be dining_entertainment']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Determine values for validation
        $profileId = (int) ($request['profile_id'] ?? $existing->profile_id);
        $nationalityId = (int) ($request['nationality_id'] ?? $existing->nationality_id);
        $adultsCount = (int) ($request['adults_count'] ?? $existing->adults_count ?? 0);
        $childrenCount = (int) ($request['children_count'] ?? $existing->children_count ?? 0);

        if ($nationalityId) {
            $countsOk = $this->validateTravelerCounts($profileId, $nationalityId, $adultsCount, $childrenCount);
            if ($countsOk !== true) {
                return ['errors' => $countsOk];
            }
        }

        if (! $canFinancial) {
            return $request;
        }

        // Refresh currency rate if currency_id provided
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        // Recompute totals
        $adultTotal = isset($request['adult_total'])
            ? (float) $request['adult_total']
            : ((float) ($request['adult_price'] ?? $existing->adult_price ?? 0)) * $adultsCount;
        $childTotal = isset($request['child_total'])
            ? (float) $request['child_total']
            : ((float) ($request['child_price'] ?? $existing->child_price ?? 0)) * $childrenCount;
        $request['adult_total'] = $adultTotal;
        $request['child_total'] = $childTotal;
        $request['grand_total'] = $adultTotal + $childTotal;

        return $request;
    }

    private function validateTravelerCounts(int $profileId, int $nationalityId, int $adultsRequested, int $childrenRequested)
    {
        $totals = ProfileTraveler::query()
            ->where('profile_id', $profileId)
            ->where('nationality_id', $nationalityId)
            ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
            ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
            ->first();

        if (! $totals || ((int) $totals->adults_total + (int) $totals->children_total) === 0) {
            return [
                'nationality_id' => ['لا يوجد مسافرون بهذه الجنسية مرتبطون بالبروفايل'],
            ];
        }

        $errors = [];
        if ($adultsRequested > (int) $totals->adults_total) {
            $errors['adults_count'][] = 'عدد الكبار المطلوب أكبر من المسجل في المسافرين';
        }
        if ($childrenRequested > (int) $totals->children_total) {
            $errors['children_count'][] = 'عدد الأطفال المطلوب أكبر من المسجل في المسافرين';
        }

        return empty($errors) ? true : $errors;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // This type has no tax persistence - dining_entertainment doesn't include taxes
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // This type has no tax persistence - dining_entertainment doesn't include taxes
    }

    private function authorizeBasic(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_RESTURANT_ENTERTAINMENT_BASIC_INFO' : 'ADD_RESTURANT_ENTERTAINMENT_BASIC_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    private function authorizeFinancial(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_RESTURANT_ENTERTAINMENT_FINANCIAL_INFO' : 'ADD_RESTURANT_ENTERTAINMENT_FINANCIAL_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }
}
