<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service;
use App\Models\General\TaxRate;
use App\Models\invoice\AirlineTickets;
use App\Models\invoice\Segments;
use App\sys\Enums\ServicesType;

class FlightsHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        return [
            'travel_tourism_type' => ['required', 'in:'.ServicesType::FLIGHTS->value],
            // profile_id اختياري: لو أُرسل يجب أن يكون صحيحاً، لو لم يُرسل نُخزِّن null
            'profile_id' => ['sometimes', 'nullable', 'integer', 'exists:pr_profile,id'],
            'pnr_number' => ['required', 'string', 'max:255'],
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'ticket_date' => ['required', 'date'],
            'domain_id' => ['nullable', 'integer'],
            'office_id' => ['nullable', 'string', 'max:50'],
            'direction_type' => ['required', 'string', 'in:one_way,round_trip'],
            // economy_class الآن يقبل جميع درجات الحجز (First/Business/Premium/Economy booking classes)
            'economy_class' => ['nullable', 'string', 'in:F,P,A,R,J,C,D,Z,I,W,E,Y,B,H,K,M,L,V,S,N,Q,O,T,X,G'],
            'segments' => ['required', 'array', 'min:1'],
            'segments.*.origin_airport' => ['required', 'integer', 'exists:airports,id'],
            'segments.*.destination_airport' => ['required', 'integer', 'exists:airports,id'],
            'segments.*.date' => ['required', 'date'],
            'segments.*.time' => ['nullable', 'string'],
            'segments.*.type' => ['nullable', 'string', 'in:go,return'],
            'segments.*.flight_number' => ['nullable', 'string', 'max:100'],
            'segments.*.flight_company' => ['nullable', 'integer', 'exists:airlines,id'],
            'pax' => ['required', 'array'],
            'pax.packs_price' => ['required', 'array'],
            'pax.packs_price.adult' => ['required', 'array'],
            'pax.packs_price.adult.count' => ['required', 'integer', 'min:0'],
            'pax.packs_price.adult.cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.adult.tax' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.adult.earned_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.adult.granted_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.adult.total_emd' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.adult.total_cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child' => ['required', 'array'],
            'pax.packs_price.child.count' => ['required', 'integer', 'min:0'],
            'pax.packs_price.child.cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child.tax' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child.earned_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child.granted_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child.total_emd' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.child.total_cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant' => ['required', 'array'],
            'pax.packs_price.infant.count' => ['required', 'integer', 'min:0'],
            'pax.packs_price.infant.cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant.tax' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant.earned_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant.granted_discount' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant.total_emd' => ['required', 'numeric', 'min:0'],
            'pax.packs_price.infant.total_cost' => ['required', 'numeric', 'min:0'],
            'pax.packs_details' => ['required', 'array', 'min:1'],
            'pax.packs_details.*.title' => ['nullable', 'string'],
            'pax.packs_details.*.first_name' => ['required', 'string', 'max:100'],
            'pax.packs_details.*.last_name' => ['required', 'string', 'max:100'],
            'pax.packs_details.*.passport_number' => ['nullable', 'string', 'max:50'],
            'pax.packs_details.*.ticket_number' => ['nullable', 'string', 'max:20'],
            'pax.packs_details.*.emd' => ['nullable', 'string', 'in:NONE,EX,ME,SE,DE,FE'],
            'pax.packs_details.*.chair_number' => ['nullable', 'string', 'max:50'],
            'pax.packs_details.*.notes' => ['nullable', 'string'],
            'pax.packs_details.*.direction_type' => ['nullable', 'string', 'in:go,back'],
            'pax.packs_details.*.type' => ['required', 'string', 'in:Adult,Child,Infant'],
            // Financial fields
            'supplier_id' => ['nullable', 'integer', 'exists:su_supplier,id'],
            'tax_rate_id' => ['nullable', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'cost' => ['nullable', 'numeric', 'min:0'],
            'currency_id' => ['nullable', 'integer', 'exists:currencies,id'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'selling_price' => ['nullable', 'numeric', 'min:0'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'is_by_handling' => ['nullable', 'boolean'],
        ];
    }

    public function getUpdateRules(array $request): array
    {
        return [
            'pnr_number' => ['sometimes', 'string', 'max:255'],
            'service_id' => ['sometimes', 'integer', 'exists:services,id'],
            'ticket_date' => ['sometimes', 'date'],
            'domain_id' => ['sometimes', 'nullable', 'integer'],
            'office_id' => ['sometimes', 'nullable', 'string', 'max:50'],
            'direction_type' => ['sometimes', 'string', 'in:one_way,round_trip'],
            // economy_class الآن يقبل جميع درجات الحجز (First/Business/Premium/Economy booking classes)
            'economy_class' => ['sometimes', 'nullable', 'string', 'in:F,P,A,R,J,C,D,Z,I,W,E,Y,B,H,K,M,L,V,S,N,Q,O,T,X,G'],
            'segments' => ['sometimes', 'array', 'min:1'],
            'segments.*.origin_airport' => ['required_with:segments', 'integer', 'exists:airports,id'],
            'segments.*.destination_airport' => ['required_with:segments', 'integer', 'exists:airports,id'],
            'segments.*.date' => ['required_with:segments', 'date'],
            'segments.*.time' => ['nullable', 'string'],
            'segments.*.type' => ['nullable', 'string', 'in:go,return'],
            'segments.*.flight_number' => ['nullable', 'string', 'max:100'],
            'segments.*.flight_company' => ['nullable', 'integer', 'exists:airlines,id'],
            'pax' => ['sometimes', 'array'],
            'pax.packs_price' => ['required_with:pax', 'array'],
            'pax.packs_details' => ['sometimes', 'array', 'min:1'],
            'pax.packs_details.*.first_name' => ['required_with:pax.packs_details', 'string', 'max:100'],
            'pax.packs_details.*.last_name' => ['required_with:pax.packs_details', 'string', 'max:100'],
            'pax.packs_details.*.type' => ['required_with:pax.packs_details', 'string', 'in:Adult,Child,Infant'],
            'supplier_id' => ['sometimes', 'nullable', 'integer', 'exists:su_supplier,id'],
            'tax_rate_id' => ['sometimes', 'nullable', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'cost' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'currency_id' => ['sometimes', 'nullable', 'integer', 'exists:currencies,id'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'selling_price' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
        ];
    }

    public function prepareForCreate(array $request): array|false
    {
        // Validate service type
        if (isset($request['service_id'])) {
            $service = Service::find($request['service_id']);
            // التحقق من نوع الخدمة
            // في جدول services العمود اسمه 'type' وليس 'travel_tourism_type'
            // travel_tourism_type موجود فقط في invoice_services
            // الآن يمكن استخدام $service->type مباشرة بعد إصلاح accessor في Model Service
            if ($service && $service->type !== ServicesType::FLIGHTS->value) {
                return ['errors' => ['service_id' => ['service type must be flights']]];
            }
        }

        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Validate and process pax data
        $paxValidation = $this->validatePaxData($request);
        if ($paxValidation !== true) {
            return $paxValidation;
        }

        // Map economy_class لضمان أن القيمة دائماً واحدة من Y,C,F,B (مع دعم القيم القديمة)
        if (isset($request['economy_class'])) {
            $request['economy_class'] = $this->mapEconomyClass($request['economy_class']);
        }

        // Handle financial fields
        // Currency rate
        $currencyId = (int) ($request['currency_id'] ?? null);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? 1;
        } else {
            $request['currency_rate'] = 1;
        }

        // Calculate grand_total from cost unless provided صراحة
        $cost = (float) ($request['cost'] ?? 0);
        // إذا أُرسلت total_tax نحترمها؛ غير ذلك نحتسبها من tax_rate_id
        if (array_key_exists('total_tax', $request)) {
            $totalTax = (float) $request['total_tax'];
        } else {
            $totalTax = 0.0;
            $taxRateIds = $request['tax_rate_id'] ?? [];
            if (is_array($taxRateIds) && ! empty($taxRateIds)) {
                $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
                $sumPercent = (float) $rates->sum();
                $totalTax = ($cost * $sumPercent) / 100.0;
            }
        }
        $request['total_tax'] = $totalTax;
        // إذا أُرسلت grand_total نحترمها، وإلا نحتسبها
        $request['grand_total'] = array_key_exists('grand_total', $request)
            ? (float) $request['grand_total']
            : $cost + $totalTax;
        $request['supplier_amount'] = (float) ($request['selling_price'] ?? 0);

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        if (array_key_exists('service_id', $request)) {
            $service = Service::find($request['service_id']);
            // التحقق من نوع الخدمة
            // في جدول services العمود اسمه 'type' وليس 'travel_tourism_type'
            // travel_tourism_type موجود فقط في invoice_services
            // الآن يمكن استخدام $service->type مباشرة بعد إصلاح accessor في Model Service
            if ($service && $service->type !== ServicesType::FLIGHTS->value) {
                return ['errors' => ['service_id' => ['service type must be flights']]];
            }
        }

        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        // Validate and process pax data if provided
        if (isset($request['pax'])) {
            $paxValidation = $this->validatePaxData($request);
            if ($paxValidation !== true) {
                return $paxValidation;
            }
        }

        // Map economy_class if provided (Y,C,F,B أو القيم النصية القديمة)
        if (isset($request['economy_class'])) {
            $request['economy_class'] = $this->mapEconomyClass($request['economy_class']);
        }

        // Handle financial fields
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        if (array_key_exists('cost', $request) || array_key_exists('tax_rate_id', $request) || array_key_exists('total_tax', $request) || array_key_exists('grand_total', $request)) {
            $cost = (float) ($request['cost'] ?? $existing->cost ?? $existing->grand_total ?? 0);

            // إذا أُرسلت total_tax نحترمها، وإلا نعيد حسابها من tax_rate_id أو نستخدم الموجودة
            if (array_key_exists('total_tax', $request)) {
                $totalTax = (float) $request['total_tax'];
            } else {
                $taxRateIds = $request['tax_rate_id'] ?? null;
                if (is_array($taxRateIds)) {
                    $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
                    $sumPercent = (float) $rates->sum();
                    $totalTax = ($cost * $sumPercent) / 100.0;
                } else {
                    $totalTax = (float) ($existing->total_tax ?? 0);
                }
            }

            $request['total_tax'] = $totalTax;
            // grand_total: إذا أُرسلت نحترمها وإلا نحتسب
            $request['grand_total'] = array_key_exists('grand_total', $request)
                ? (float) $request['grand_total']
                : $cost + $totalTax;
        }

        if (array_key_exists('selling_price', $request)) {
            $request['supplier_amount'] = (float) ($request['selling_price'] ?? $existing->supplier_amount ?? 0);
        }

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // Save segments
        if (isset($request['segments']) && is_array($request['segments'])) {
            $this->saveSegments($invoiceService->id, $request['segments']);
        }

        // Save airline tickets (pax.packs_details)
        if (isset($request['pax']['packs_details']) && is_array($request['pax']['packs_details'])) {
            $this->saveAirlineTickets($invoiceService, $request);
        }

        // Save taxes
        if (isset($request['tax_rate_id']) && is_array($request['tax_rate_id'])) {
            $this->saveTaxes($invoiceService, $request);
        }
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // Update segments if provided
        if (array_key_exists('segments', $request)) {
            // Delete existing segments
            Segments::where('invoice_services_id', $invoiceService->id)->delete();
            // Save new segments
            if (is_array($request['segments']) && ! empty($request['segments'])) {
                $this->saveSegments($invoiceService->id, $request['segments']);
            }
        }

        // Update airline tickets if provided
        if (array_key_exists('pax', $request) && isset($request['pax']['packs_details'])) {
            // Delete existing tickets
            AirlineTickets::where('invoice_service_id', $invoiceService->id)->delete();
            // Save new tickets
            if (is_array($request['pax']['packs_details']) && ! empty($request['pax']['packs_details'])) {
                $this->saveAirlineTickets($invoiceService, $request);
            }
        }

        // Update taxes if provided
        if (array_key_exists('tax_rate_id', $request)) {
            $this->saveTaxes($invoiceService, $request, true);
        }
    }

    private function saveSegments(int $invoiceServiceId, array $segments): void
    {
        foreach ($segments as $segment) {
            Segments::create([
                'invoice_services_id' => $invoiceServiceId,
                'origin_airport' => $segment['origin_airport'] ?? null,
                'destination_airport' => $segment['destination_airport'] ?? null,
                'date' => $segment['date'] ?? null,
                'time' => $segment['time'] ?? null,
                'type' => $segment['type'] ?? 'go',
                'flight_number' => $segment['flight_number'] ?? null,
                'flight_company' => $segment['flight_company'] ?? null,
            ]);
        }
    }

    private function saveAirlineTickets(object $invoiceService, array $request): void
    {
        $packsPrice = $request['pax']['packs_price'] ?? [];
        $packsDetails = $request['pax']['packs_details'] ?? [];
        $directionType = $request['direction_type'] ?? 'one_way';
        $isRoundTrip = $directionType === 'round_trip';

        // Calculate per-person costs from packs_price
        // الأسعار في packs_price هي الإجمالي لكل نوع مسافر، نقسمها على count للحصول على السعر لكل مسافر
        $adultCostPerPerson = $this->calculatePerPersonCost($packsPrice['adult'] ?? []);
        $childCostPerPerson = $this->calculatePerPersonCost($packsPrice['child'] ?? []);
        $infantCostPerPerson = $this->calculatePerPersonCost($packsPrice['infant'] ?? []);

        // في حالة الرحلة ذهاب وعودة، نقسم السعر على 2 للحصول على السعر لكل تذكرة (go أو back)
        // في حالة الرحلة ذهاب فقط، السعر لكل تذكرة = السعر لكل مسافر
        $ticketDivisor = $isRoundTrip ? 2 : 1;

        $adultCostPerTicket = [
            'cost' => $adultCostPerPerson['cost'] / $ticketDivisor,
            'tax' => $adultCostPerPerson['tax'] / $ticketDivisor,
            'earned_discount' => $adultCostPerPerson['earned_discount'] / $ticketDivisor,
            'granted_discount' => $adultCostPerPerson['granted_discount'] / $ticketDivisor,
        ];

        $childCostPerTicket = [
            'cost' => $childCostPerPerson['cost'] / $ticketDivisor,
            'tax' => $childCostPerPerson['tax'] / $ticketDivisor,
            'earned_discount' => $childCostPerPerson['earned_discount'] / $ticketDivisor,
            'granted_discount' => $childCostPerPerson['granted_discount'] / $ticketDivisor,
        ];

        $infantCostPerTicket = [
            'cost' => $infantCostPerPerson['cost'] / $ticketDivisor,
            'tax' => $infantCostPerPerson['tax'] / $ticketDivisor,
            'earned_discount' => $infantCostPerPerson['earned_discount'] / $ticketDivisor,
            'granted_discount' => $infantCostPerPerson['granted_discount'] / $ticketDivisor,
        ];

        $totalEmd = (float) ($packsPrice['adult']['total_emd'] ?? 0) +
                   (float) ($packsPrice['child']['total_emd'] ?? 0) +
                   (float) ($packsPrice['infant']['total_emd'] ?? 0);

        $usedEmd = 0.0;

        // Group passengers by their base info (without direction)
        $passengersByBase = [];
        foreach ($packsDetails as $detail) {
            $key = ($detail['first_name'] ?? '').'|'.($detail['last_name'] ?? '').'|'.($detail['type'] ?? 'Adult');
            if (! isset($passengersByBase[$key])) {
                $passengersByBase[$key] = $detail;
            }
        }

        foreach ($passengersByBase as $baseDetail) {
            $passengerType = strtoupper($baseDetail['type'] ?? 'ADULT');
            $costData = match ($passengerType) {
                'ADULT' => $adultCostPerTicket,
                'CHILD' => $childCostPerTicket,
                'INFANT' => $infantCostPerTicket,
                default => ['cost' => 0, 'tax' => 0, 'earned_discount' => 0, 'granted_discount' => 0],
            };

            $directions = $isRoundTrip ? ['go', 'back'] : ['go'];

            foreach ($directions as $direction) {
                // Find detail with matching direction_type or use base detail
                $detail = $baseDetail;
                foreach ($packsDetails as $d) {
                    if (($d['first_name'] ?? '') === ($baseDetail['first_name'] ?? '') &&
                        ($d['last_name'] ?? '') === ($baseDetail['last_name'] ?? '') &&
                        ($d['type'] ?? '') === ($baseDetail['type'] ?? '') &&
                        ($d['direction_type'] ?? 'go') === $direction) {
                        $detail = $d;
                        break;
                    }
                }

                // Calculate EMD for this ticket
                // EMD يتم توزيعه على التذاكر، لذا نقسمه على عدد التذاكر
                $emdPrice = 0.0;
                if (isset($detail['emd']) && $detail['emd'] !== 'NONE' && $totalEmd > $usedEmd) {
                    // Distribute EMD - simple approach: use remaining EMD if available
                    $remainingEmd = $totalEmd - $usedEmd;
                    // EMD لكل تذكرة = EMD الإجمالي / عدد التذاكر
                    $emdPerTicket = $totalEmd / max(1, count($packsDetails));
                    $emdPrice = min($remainingEmd, $emdPerTicket, $costData['cost'] * 0.1); // Max 10% of cost as EMD
                    $usedEmd += $emdPrice;
                }

                $ticketData = [
                    'invoice_service_id' => $invoiceService->id,
                    'company_id' => $invoiceService->company_id ?? null,
                    'month' => date('Y-m', strtotime($request['ticket_date'] ?? 'now')),
                    'issue_date' => $request['ticket_date'] ?? now(),
                    'chair_num' => $detail['chair_number'] ?? null,
                    // flight_num من جدول airline_tickets (سيتم تحديثه من الـ segment لاحقاً إن وجد)
                    'flight_num' => $detail['ticket_number'] ?? null,
                    'from_airline_id' => null, // Will be set from segments if needed
                    'to_airline_id' => null,
                    'go_back' => $direction,
                    'from_airport_id' => null, // Will be set from segments
                    'to_airport_id' => null,
                    'ticket_num' => $detail['ticket_number'] ?? null,
                    'passport_num' => $detail['passport_number'] ?? null,
                    'title' => $detail['title'] ?? null,
                    // نستخدم القيمة كما في invoice_services، مع تحويلها إلى Y/C/F/B عند الحاجة
                    'economy_class' => $this->mapEconomyClassToEnum($request['economy_class'] ?? 'Y'),
                    'passenger_type' => $passengerType,
                    // نخزن الاسم بدون اللقب، مع فصل اللقب في حقل مستقل
                    'passenger_name' => trim(($detail['first_name'] ?? '').' '.($detail['last_name'] ?? '')),
                    'ticket_emd' => $detail['emd'] ?? 'NONE',
                    'emd_price' => $emdPrice,
                    'fare_price' => $costData['cost'], // السعر لكل تذكرة
                    'tax' => $costData['tax'], // الضرائب لكل تذكرة
                    'earned_discount' => $costData['earned_discount'], // الخصم المكتسب لكل تذكرة
                    'granted_discount' => $costData['granted_discount'], // الخصم الممنوح لكل تذكرة
                    'total_price' => $costData['cost'] + $costData['tax'] - $costData['earned_discount'] - $costData['granted_discount'], // السعر الإجمالي لكل تذكرة
                    'pnr' => $request['pnr_number'] ?? null,
                    'note' => $detail['notes'] ?? null,
                    'created' => now(),
                    'modified' => now(),
                ];

                // Try to match with segments for airport and airline info
                if (isset($request['segments']) && is_array($request['segments'])) {
                    $segment = $this->findSegmentForDirection($request['segments'], $direction);
                    if ($segment) {
                        $ticketData['from_airport_id'] = $segment['origin_airport'] ?? null;
                        $ticketData['to_airport_id'] = $segment['destination_airport'] ?? null;
                        $ticketData['from_airline_id'] = $segment['flight_company'] ?? null;
                        $ticketData['to_airline_id'] = $segment['flight_company'] ?? null;
                        $ticketData['flight_num'] = $segment['flight_number'] ?? null;
                    }
                }

                AirlineTickets::create($ticketData);
            }
        }
    }

    private function findSegmentForDirection(array $segments, string $direction): ?array
    {
        foreach ($segments as $segment) {
            $segmentType = $segment['type'] ?? 'go';
            if (($direction === 'go' && $segmentType === 'go') ||
                ($direction === 'back' && $segmentType === 'return')) {
                return $segment;
            }
        }

        // If no match, return first segment
        return $segments[0] ?? null;
    }

    private function calculatePerPersonCost(array $packPrice): array
    {
        $count = max(1, (int) ($packPrice['count'] ?? 1));

        return [
            'cost' => ((float) ($packPrice['cost'] ?? 0)) / $count,
            'tax' => ((float) ($packPrice['tax'] ?? 0)) / $count,
            'earned_discount' => ((float) ($packPrice['earned_discount'] ?? 0)) / $count,
            'granted_discount' => ((float) ($packPrice['granted_discount'] ?? 0)) / $count,
        ];
    }

    private function validatePaxData(array $request): array|bool
    {
        if (! isset($request['pax']) || ! is_array($request['pax'])) {
            return ['errors' => ['pax' => ['pax data is required']]];
        }

        $packsPrice = $request['pax']['packs_price'] ?? [];
        $packsDetails = $request['pax']['packs_details'] ?? [];
        $directionType = $request['direction_type'] ?? 'one_way';
        $isRoundTrip = $directionType === 'round_trip';

        // Validate counts match
        $adultCount = (int) ($packsPrice['adult']['count'] ?? 0);
        $childCount = (int) ($packsPrice['child']['count'] ?? 0);
        $infantCount = (int) ($packsPrice['infant']['count'] ?? 0);
        $totalExpected = $adultCount + $childCount + $infantCount;

        // Count actual passengers in packs_details
        $actualCount = count($packsDetails);
        $expectedWithRoundTrip = $isRoundTrip ? $totalExpected * 2 : $totalExpected;

        if ($actualCount !== $expectedWithRoundTrip) {
            return ['errors' => ['pax' => ['Passenger count mismatch. Expected: '.$expectedWithRoundTrip.', Got: '.$actualCount]]];
        }

        // Validate EMD usage
        $totalEmd = (float) ($packsPrice['adult']['total_emd'] ?? 0) +
                   (float) ($packsPrice['child']['total_emd'] ?? 0) +
                   (float) ($packsPrice['infant']['total_emd'] ?? 0);

        if ($totalEmd > 0) {
            $usedEmd = 0;
            foreach ($packsDetails as $detail) {
                if (isset($detail['emd']) && $detail['emd'] !== 'NONE') {
                    // Simple validation: EMD should be used
                    $usedEmd++;
                }
            }
            if ($usedEmd === 0 && $totalEmd > 0) {
                return ['errors' => ['pax' => ['EMD value provided but not used in any ticket']]];
            }
        }

        return true;
    }

    /**
     * Normalize economy_class to one of the supported booking classes.
     *
     * Supported codes:
     * - First:   F, P, A, R
     * - Business: J, C, D, Z, I
     * - Premium: W, E
     * - Economy: Y, B, H, K, M, L, V, S, N, Q, O, T, X, G
     */
    private function mapEconomyClass(string $class): string
    {
        $upper = strtoupper($class);

        $allowed = [
            // First
            'F', 'P', 'A', 'R',
            // Business
            'J', 'C', 'D', 'Z', 'I',
            // Premium Economy
            'W', 'E',
            // Economy
            'Y', 'B', 'H', 'K', 'M', 'L', 'V', 'S', 'N', 'Q', 'O', 'T', 'X', 'G',
        ];

        // إذا كانت القيمة بالفعل كود صالح نرجعها كما هي
        if (in_array($upper, $allowed, true)) {
            return $upper;
        }

        // دعم القيم النصية القديمة كخلفية
        return match (strtolower($class)) {
            'first' => 'F',
            'business' => 'J',
            'premium_economy' => 'W',
            'economy' => 'Y',
            default => 'Y',
        };
    }

    private function mapEconomyClassToEnum(string $class): string
    {
        return $this->mapEconomyClass($class);
    }

    private function saveTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (! is_array($taxRateIds)) {
            return;
        }

        if ($replace) {
            \App\Models\invoice\InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $cost = (float) ($request['cost'] ?? $invoiceService->grand_total ?? 0);

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($cost * (float) $rate->percentage) / 100.0;
            \App\Models\invoice\InvoiceServicesTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'percentage' => (float) $rate->percentage,
                'tax_amount' => $amount,
            ]);
        }
    }
}
