<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\sys\Enums\ServicesType;
use Illuminate\Support\Facades\Gate;

class TourRepsHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'add');
        $financial = $this->authorizeFinancial($request, 'add');

        return [
            'service_id' => [$basic, 'integer', 'exists:services,id'],
            'service_provider' => [$financial, 'in:supplier_id,guide_id'],
            // financial fields optional but required when present
            'supplier_id' => ['sometimes', $financial, 'integer', 'exists:su_supplier,id'],
            'guide_id' => ['sometimes', $financial, 'integer', 'exists:users,id'],
            'extra_cost' => ['sometimes', $financial, 'numeric', 'min:0'],
            'daily_cost' => ['sometimes', $financial, 'numeric', 'min:0'],
            'tip_amount' => ['sometimes', $financial, 'numeric', 'min:0'],
            'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'executive_id' => ['sometimes', $financial, 'integer', 'exists:users,id'],
            'guide_language_id' => ['nullable', 'integer', 'exists:guide_language,id'],
            'people_count' => [$basic, 'integer', 'min:0'],
            'notes' => ['nullable', 'string', 'max:1000'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'daily_program_id' => [$basic, 'integer', 'exists:pr_daily_programs,id'],

        ];
    }

    public function getUpdateRules(array $request): array
    {
        $basic = $this->authorizeBasic($request, 'edit');
        $financial = $this->authorizeFinancial($request, 'edit');

        $rules = [
            'total_tips' => 'prohibited',
            'grand_total' => 'prohibited',
            'currency_rate' => 'prohibited',
            'is_by_handling' => 'nullabel|boolean',
        ];

        if (array_key_exists('service_id', $request)) {
            $rules['service_id'] = [$basic, 'integer', 'exists:services,id'];
        }
        if (array_key_exists('service_provider', $request)) {
            $rules['service_provider'] = ['required', 'in:supplier_id,guide_id'];
        }
        if (array_key_exists('supplier_id', $request)) {
            $rules['supplier_id'] = [$financial, 'integer', 'exists:su_supplier,id'];
        }
        if (array_key_exists('guide_id', $request)) {
            $rules['guide_id'] = [$financial, 'integer', 'exists:users,id'];
        }
        if (array_key_exists('guide_language_id', $request)) {
            $rules['guide_language_id'] = [$basic, 'integer', 'exists:languages,id'];
        }
        if (array_key_exists('people_count', $request)) {
            $rules['people_count'] = [$basic, 'integer', 'min:0'];
        }
        if (array_key_exists('daily_cost', $request)) {
            $rules['daily_cost'] = [$financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('tip_amount', $request)) {
            $rules['tip_amount'] = [$financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('extra_cost', $request)) {
            $rules['extra_cost'] = ['nullable', $financial, 'numeric', 'min:0'];
        }
        if (array_key_exists('executive_id', $request)) {
            $rules['executive_id'] = [$financial, 'integer', 'exists:users,id'];
        }
        if (array_key_exists('currency_id', $request)) {
            $rules['currency_id'] = [$financial, 'integer', 'exists:currencies,id'];
        }
        if (array_key_exists('notes', $request)) {
            $rules['notes'] = ['nullable', 'string', 'max:1000'];
        }
        if (array_key_exists('profile_id', $request)) {
            $rules['profile_id'] = ['required', 'integer', 'exists:pr_profile,id'];
        }
        if (array_key_exists('daily_program_id', $request)) {
            $rules['daily_program_id'] = [$basic, 'integer', 'exists:pr_daily_programs,id'];
        }

        if (array_key_exists('is_by_handling', $request)) {
            $rules['is_by_handling'] = ['nullable', 'boolean'];
        }

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        $this->authorizeBasic($request, 'add');
        $this->authorizeFinancial($request, 'add');
        $canFinancial = Gate::allows('ADD_TOUR_REPO_FINANCIAL_INFO');

        // Normalize provider: if service_provider specified, nullify the other id
        $provider = $request['service_provider'] ?? null;
        if ($provider == 'supplier_id') {
            $request['guide_id'] = null;
        } elseif ($provider == 'guide_id') {
            $request['supplier_id'] = null;
            // keep supplier required; do not nullify supplier per business rules
        }
        // Validate service type
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
            return ['errors' => ['service_id' => ['service type must be tour_reps']]];
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        if (! $canFinancial) {
            $request['currency_rate'] = 1;
            $request['total_tips'] = 0;
            $request['grand_total'] = 0;

            return $request;
        }

        // Set currency rate
        $currency = isset($request['currency_id']) ? Currency::find($request['currency_id']) : null;
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Compute totals
        $peopleCount = (int) ($request['people_count'] ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $tipAmount = (float) ($request['tip_amount'] ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        $this->authorizeBasic($request, 'edit');
        $this->authorizeFinancial($request, 'edit');
        $canFinancial = Gate::allows('EDIT_TOUR_REPO_FINANCIAL_INFO');

        // Normalize provider switching on update
        if (array_key_exists('service_provider', $request)) {
            if ($request['service_provider'] === 'supplier_id') {
                $request['guide_id'] = null;
            }
        } else {
            if (array_key_exists('supplier_id', $request) && ! is_null($request['supplier_id'])) {
                $request['guide_id'] = null;
            }
        }
        // If service_id provided, ensure it's correct type
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::TOUR_REPS->value) {
                return ['errors' => ['service_id' => ['service type must be tour_reps']]];
            }
        }
        $request['is_by_handling'] = (bool) ($request['is_by_handling'] ?? false);

        if (! $canFinancial) {
            return $request;
        }

        // Refresh currency rate if currency_id provided
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        // Use provided values or fallback to existing for calculations
        $peopleCount = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $tipAmount = (float) ($request['tip_amount'] ?? $existing->tip_amount ?? 0);
        $extraCost = (float) ($request['extra_cost'] ?? $existing->extra_cost ?? 0);

        $totalTips = $tipAmount * $peopleCount;
        $grandTotal = $dailyCost + $totalTips + $extraCost;

        $request['total_tips'] = $totalTips;
        $request['grand_total'] = $grandTotal;

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // No tax persistence for tour_reps
    }

    /**
     * Authorize adding/editing basic fields (service_id, guide_language_id, people_count)
     */
    private function authorizeBasic(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_TOUR_REPO_BASIC_INFO' : 'ADD_TOUR_REPO_BASIC_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    /**
     * Authorize adding/editing financial fields (provider/supplier/guide/costs/tips/currency/extra)
     */
    private function authorizeFinancial(array $request, string $mode): string
    {
        $ability = $mode === 'edit' ? 'EDIT_TOUR_REPO_FINANCIAL_INFO' : 'ADD_TOUR_REPO_FINANCIAL_INFO';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }
}
