<?php

namespace App\sys\Services\Profile;

use App\Models\General\Currency;
use App\Models\General\TaxRate;
use App\Models\invoice\InvoiceServices;
use App\Models\invoice\InvoiceServicesTax;
use App\Models\Profile\AccommodationReservation;
use App\Models\Profile\AccommodationReservationRoom;
use App\Models\Profile\AccommodationReservationRoomTax;
use App\Models\Profile\Profile;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Repository\Profile\AccommodationReservationPriceRepository;
use App\sys\Repository\Profile\AccommodationReservationRepository;
use App\sys\Repository\Profile\AccommodationReservationRoomRepository;
use App\sys\Services;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;

class AccommodationReservationRoomService extends Services
{
    protected AccommodationReservationRoomRepository $roomRepository;

    protected AccommodationReservationRepository $reservationRepository;

    protected AccommodationReservationPriceRepository $priceRepository;

    protected InvoiceServicesRepository $invoiceServicesRepository;

    public function __construct(
        AccommodationReservationRoomRepository $roomRepository,
        AccommodationReservationRepository $reservationRepository,
        AccommodationReservationPriceRepository $priceRepository,
        InvoiceServicesRepository $invoiceServicesRepository
    ) {
        $this->roomRepository = $roomRepository;
        $this->reservationRepository = $reservationRepository;
        $this->priceRepository = $priceRepository;
        $this->invoiceServicesRepository = $invoiceServicesRepository;
    }

    public function getAll()
    {
        return $this->roomRepository->getAll();
    }

    public function getById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation_rooms,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->roomRepository->findByIdOrFail($id);

        $item->load(['currency', 'taxRate.tax', 'nationality']);

        return $item;
    }

    public function getAllByReservationId(int $id)
    {
        $rules = [
            'id' => ['required', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->roomRepository->getAllByReservationId($id);
    }

    private function computeNights(string $checkIn, string $checkOut): int
    {
        $in = Carbon::parse($checkIn);
        $out = Carbon::parse($checkOut);
        $diff = $in->diffInDays($out);

        return max(1, $diff);
    }

    private function computeFinancials(array $data, int $nights): array
    {
        $adultCount = (int) ($data['adult_count'] ?? 0);
        $childCount = (int) ($data['childreen_count'] ?? 0);
        $extraBedCount = (int) ($data['extra_bed'] ?? 0);
        $roomCount = (int) ($data['room_number'] ?? 1);

        $adultPrice = (float) ($data['adult_price'] ?? 0);
        $childPrice = (float) ($data['child_price'] ?? 0);
        $extraBedPrice = (float) ($data['extra_bed_price'] ?? 0);
        $commission = (float) ($data['commission'] ?? 0);
        $taxRatePrice = 0;
        $discount = (float) ($data['discount'] ?? 0);
        $discountType = $data['discount_type'] ?? 'value'; // 'value' | 'percentage'

        // Treat room_number as the count of rooms to multiply nightly charges
        $basePerNight = (($adultCount * $adultPrice) + ($childCount * $childPrice) + ($extraBedCount * $extraBedPrice)) * max(1, $roomCount);
        $perNightPrice = $basePerNight + ($commission * max(1, $roomCount)) + $taxRatePrice;
        $totalBeforeDiscount = $perNightPrice * $nights;

        $totalDiscount = 0.0;
        if ($discount > 0) {
            if ($discountType === 'percentage') {
                $totalDiscount = $totalBeforeDiscount * (min($discount, 100) / 100.0);
            } elseif ($discountType === 'value') {
                // Value discounts scale by room count as well
                $totalDiscount = min($discount * max(1, $roomCount), $totalBeforeDiscount);
            }
        }
        $totalAmount = max(0.0, $totalBeforeDiscount - $totalDiscount);

        return [
            'per_night_price' => $perNightPrice,
            'total_amount' => $totalAmount,
        ];
    }

    private function sumRoomsTotalsByCurrency(int $reservationId, int $currencyId): float
    {
        return (float) AccommodationReservationRoom::query()
            ->where('reservation_id', $reservationId)
            ->where('currency_id', $currencyId)
            ->sum('total_amount');
    }

    private function sumRoomTaxesByCurrency(int $reservationId, int $currencyId): array
    {
        $rows = AccommodationReservationRoomTax::query()
            ->where('accommodation_reservation_id', $reservationId)
            ->where('currency_id', $currencyId)
            ->select('tax_rate_id', DB::raw('SUM(tax_amount) as total'))
            ->groupBy('tax_rate_id')
            ->get();

        $total = (float) $rows->sum('total');
        $grouped = [];
        foreach ($rows as $row) {
            $grouped[(int) $row->tax_rate_id] = (float) $row->total;
        }

        return [$total, $grouped];
    }

    private function upsertInvoiceForReservationCurrency(
        int $reservationId,
        int $profileId,
        int $currencyId,
        ?int $supplierId,
        int $roomId,
        float $roomTotal,
        float $roomTaxTotal,
        array $roomTaxGrouped,
        ?int $invoiceServicesId = null
    ): ?InvoiceServices {
        $currency = Currency::find($currencyId);
        $rate = (float) ($currency->exchange_rate ?? 1);

        // choose target invoice (one per reservation/currency without invoice_id)
        $query = InvoiceServices::where('reservation_model', 'AccommodationReservation')
            ->where('reservation_id', $reservationId)
            ->where('currency_id', $currencyId)
            ->whereNull('invoice_id');

        if ($invoiceServicesId) {
            $query->where('id', $invoiceServicesId);
        }

        $invoice = $query->first();

        // If existing invoice is tied to a final invoice, create isolated one for this room only
        $useRoomOnlyTotals = false;
        if ($invoice && $invoice->invoice_id) {
            $invoice = null;
            $useRoomOnlyTotals = true;
        }

        if (! $invoice) {
            $totalAmount = $useRoomOnlyTotals
                ? $roomTotal
                : $this->sumRoomsTotalsByCurrency($reservationId, $currencyId);
            [$totalTax, $taxGrouped] = $useRoomOnlyTotals
                ? [$roomTaxTotal, $roomTaxGrouped]
                : $this->sumRoomTaxesByCurrency($reservationId, $currencyId);
            $grandTotal = $totalAmount + $totalTax;

            $invoice = $this->invoiceServicesRepository->create([
                'travel_tourism_type' => 'accommodation',
                'type' => 'credit',
                'purchase_price' => $totalAmount,
                'purchase_commission' => 0,
                'purchase_currency_id' => $currencyId,
                'currency_id' => $currencyId,
                'currency_rate' => $rate,
                'purchase_rate' => $rate,
                'purchase_tax_amount' => $totalTax,
                'purchase_tax_rate' => $rate,
                'total_tax' => $totalTax,
                'grand_total' => $grandTotal,
                'remaining_to_supplier' => $grandTotal,
                'supplier_id' => $supplierId,
                'profile_id' => $profileId,
                'reservation_model' => 'AccommodationReservation',
                'reservation_id' => $reservationId,
                'is_by_handling' => $useRoomOnlyTotals ? 1 : 0,
            ]);

            // sync taxes
            InvoiceServicesTax::where('invoice_services_id', $invoice->id)->delete();
            foreach ($taxGrouped as $taxRateId => $amount) {
                $taxRate = TaxRate::find($taxRateId);
                InvoiceServicesTax::create([
                    'invoice_services_id' => $invoice->id,
                    'tax_rate_id' => $taxRateId,
                    'percentage' => (float) ($taxRate?->percentage ?? 0),
                    'tax_amount' => $amount,
                ]);
            }
        } else {
            // aggregate totals including this room and any others on same reservation/currency
            $totalAmount = $this->sumRoomsTotalsByCurrency($reservationId, $currencyId);
            [$totalTax, $taxGrouped] = $this->sumRoomTaxesByCurrency($reservationId, $currencyId);
            $grandTotal = $totalAmount + $totalTax;

            $paidToSupplier = (float) ($invoice->paid_to_supplier ?? 0);
            $remaining = max(0, $grandTotal - $paidToSupplier);
            $this->invoiceServicesRepository->update($invoice, [
                'purchase_price' => $totalAmount,
                'purchase_tax_amount' => $totalTax,
                'purchase_currency_id' => $currencyId,
                'purchase_tax_rate' => $rate,
                'purchase_rate' => $rate,
                'currency_id' => $currencyId,
                'currency_rate' => $rate,
                'total_tax' => $totalTax,
                'grand_total' => $grandTotal,
                'remaining_to_supplier' => $remaining,
                'supplier_id' => $supplierId,
                'profile_id' => $profileId,
                'is_by_handling' => $invoice->is_by_handling ?? 0,
            ]);

            // sync taxes
            InvoiceServicesTax::where('invoice_services_id', $invoice->id)->delete();
            foreach ($taxGrouped as $taxRateId => $amount) {
                $taxRate = TaxRate::find($taxRateId);
                InvoiceServicesTax::create([
                    'invoice_services_id' => $invoice->id,
                    'tax_rate_id' => $taxRateId,
                    'percentage' => (float) ($taxRate?->percentage ?? 0),
                    'tax_amount' => $amount,
                ]);
            }
        }

        // attach only current room
        AccommodationReservationRoom::where('id', $roomId)->update(['invoice_services_id' => $invoice->id]);

        return $invoice;
    }

    public function create(array $request)
    {
        $basic = $this->authorizeBasic('add');
        $financial = $this->authorizeFinancial('add');

        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'reservation_id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
            'nationality_id' => ['nullable', 'integer', 'exists:nationalities,id'],
            'check_in' => [$basic, 'date'],
            'check_out' => [$basic, 'date', 'after:check_in'],
            'adult_count' => [$basic, 'integer', 'min:0'],
            'childreen_count' => [$basic, 'integer', 'min:0'],
            'no_fee_children_count' => [$basic, 'integer', 'min:0'],
            'room_number' => ['nullable', 'integer', 'min:1'],
            'booking_date' => ['nullable', 'date'],
            'reservation_date' => ['nullable', 'date'],
            'meal_plan_id' => ['nullable', 'integer', 'exists:ac_meal_plans,id'],
            'bed_type' => ['nullable', 'in:king,queen,single,double,sofa'],
            'room_type_id' => ['nullable', 'integer', 'exists:ac_room_types,id'],
            'room_view_id' => ['nullable', 'integer', 'exists:ac_room_views,id'],
            'extra_bed' => ['nullable', 'integer', 'min:0'],
            'currency_id' => [$financial, 'integer', 'exists:currencies,id'],
            'adult_price' => [$financial, 'numeric', 'min:0'],
            'child_price' => [$financial, 'numeric', 'min:0'],
            'extra_bed_price' => ['nullable', 'numeric', 'min:0'],
            'commission' => ['nullable', 'numeric', 'min:0'],
            'tax_rate_id' => ['nullable', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'tax_rate_price' => ['nullable', 'numeric', 'min:0'],
            'discount' => ['nullable', 'numeric', 'min:0'],
            'discount_type' => ['nullable', 'in:value,percentage'],

        ];

        $validator = Validator::make($request, $rules);
        $validator->after(function ($v) use ($request) {
            if (($request['discount_type'] ?? null) === 'percentage' && ($request['discount'] ?? null) !== null) {
                if ((float) $request['discount'] > 100) {
                    $v->errors()->add('discount', 'When discount_type is percentage, discount must be <= 100');
                }
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // capacity validation by nationality when provided (adults/children only)
        $reservationId = (int) $request['reservation_id'];
        $profileId = (int) $request['profile_id'];
        $nationalityId = (int) ($request['nationality_id'] ?? 0);
        if ($nationalityId > 0) {
            $totals = $this->roomRepository->getTotalsByReservationProfileNationality($reservationId, $profileId, $nationalityId);
            $currentAdults = (int) ($totals->adults ?? 0);
            $currentChildren = (int) ($totals->children ?? 0);
            $currentNoFee = 0;
        } else {
            $totals = $this->roomRepository->getTotalsByReservationProfile($reservationId, $profileId);
            $currentAdults = (int) ($totals->adults ?? 0);
            $currentChildren = (int) ($totals->children ?? 0);
            $currentNoFee = (int) ($totals->no_fee_children ?? 0);
        }

        $newAdults = (int) $request['adult_count'];
        $newChildren = (int) $request['childreen_count'];
        $newNoFee = (int) ($request['no_fee_children_count'] ?? 0);

        $profile = (new \App\sys\Repository\Profile\ProfileRepository)->findByIdOrFail((int) $request['profile_id']);
        if ($nationalityId > 0) {
            $travelerTotals = \App\Models\Profile\ProfileTraveler::query()
                ->where('profile_id', $profileId)
                ->where('nationality_id', $nationalityId)
                ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
                ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
                ->first();
            $profileAdults = (int) ($travelerTotals->adults_total ?? 0);
            $profileChildren = (int) ($travelerTotals->children_total ?? 0);
            $profileNoFee = PHP_INT_MAX; // ignore no-fee children caps in nationality mode
        } else {
            $profileAdults = (int) $profile->adults_count;
            $profileChildren = (int) $profile->children_count;
            $profileNoFee = (int) $profile->children_no_fee_count;
        }

        if ($currentAdults + $newAdults > $profileAdults) {
            $this->setError(['adult_count' => ['Exceeds profile adults_count']]);

            return false;
        }
        if ($currentChildren + $newChildren > $profileChildren) {
            $this->setError(['childreen_count' => ['Exceeds profile children_count']]);

            return false;
        }
        if ($currentNoFee + $newNoFee > $profileNoFee) {
            $this->setError(['no_fee_children_count' => ['Exceeds profile children_no_fee_count']]);

            return false;
        }

        // Map optional reservation_date to booking_date if provided
        if (! empty($request['reservation_date']) && empty($request['booking_date'])) {
            $request['booking_date'] = $request['reservation_date'];
        }

        $nights = $this->computeNights($request['check_in'], $request['check_out']);
        $canFinancial = Gate::allows('ADD_ROOM_COST');
        $financials = $canFinancial ? $this->computeFinancials($request, $nights) : ['per_night_price' => 0, 'total_amount' => 0];
        $request['nights_count'] = $nights;
        $request = array_merge($request, $financials);

        $room = $this->roomRepository->create($request);

        // start calclaute prices
        if ($canFinancial) {
            // Tax calculation and persistence per selected tax rates
            $currency = Currency::find((int) $request['currency_id']);
            $exchangeRate = (float) ($currency->exchange_rate ?? 1);
            $taxIds = $request['tax_rate_id'] ?? [];
            if (! is_array($taxIds)) {
                $taxIds = [$taxIds];
            }
            $taxRates = empty($taxIds) ? collect() : TaxRate::whereIn('id', $taxIds)->get();
            $totalTaxAmount = 0.0;
            // remove any existing rows just in case (fresh create should have none)
            AccommodationReservationRoomTax::where('accommodation_reservation_rooms_id', (int) $room->id)->delete();
            foreach ($taxRates as $taxRate) {
                $calculatedAmount = 0.0;
                if (strtolower((string) $taxRate->type) === 'general') {
                    $percentage = (float) $taxRate->percentage;
                    $calculatedAmount = ((float) $room->total_amount) * ($percentage / 100.0);
                } elseif (strtolower((string) $taxRate->type) === 'revenue') {
                    $calculatedAmount = 0.0;
                }

                AccommodationReservationRoomTax::create([
                    'tax_rate_id' => (int) $taxRate->id,
                    'tax_amount' => $calculatedAmount,
                    'tax_rate_amount' => $exchangeRate,
                    'currency_id' => (int) $request['currency_id'],
                    'accommodation_reservation_rooms_id' => (int) $room->id,
                    'accommodation_reservation_id' => (int) $request['reservation_id'],
                    'profile_id' => (int) $request['profile_id'],
                ]);
                $totalTaxAmount += $calculatedAmount;
            }

            // Persist per-night tax on the room row
            $taxPerNight = $nights > 0 ? ($totalTaxAmount / $nights) : 0.0;
            $room->tax_rate_price = $taxPerNight;
            $room->save();

            // aggregate per reservation / currency into invoice_services
            $rese = AccommodationReservation::find((int) $request['reservation_id']);
            $invoice = $this->upsertInvoiceForReservationCurrency(
                (int) $request['reservation_id'],
                (int) $request['profile_id'],
                (int) $request['currency_id'],
                $rese?->supplier_id,
                (int) $room->id,
                (float) $room->total_amount,
                $totalTaxAmount,
                $taxRates->mapWithKeys(fn ($t) => [(int) $t->id => (float) ($t->type === 'general' ? $room->total_amount * ((float) $t->percentage / 100.0) : 0)])->toArray()
            );
            if ($invoice) {
                $room->invoice_services_id = $invoice->id;
                $room->save();
            }
        }

        return $room;
    }

    public function update(array $request)
    {
        $basic = $this->authorizeBasic('edit');
        $financial = $this->authorizeFinancial('edit');

        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation_rooms,id'],
            'profile_id' => ['sometimes', 'integer', 'exists:pr_profile,id'],
            'reservation_id' => ['sometimes', 'integer', 'exists:pr_accommodation_reservation,id'],
            'nationality_id' => ['sometimes', 'nullable', 'integer', 'exists:nationalities,id'],
            'check_in' => ['sometimes', $basic, 'date'],
            'check_out' => ['sometimes', $basic, 'date'],
            'adult_count' => ['sometimes', $basic, 'integer', 'min:0'],
            'childreen_count' => ['sometimes', $basic, 'integer', 'min:0'],
            'no_fee_children_count' => ['sometimes', $basic, 'integer', 'min:0'],
            'room_number' => ['sometimes', 'nullable', 'integer', 'min:1'],
            'booking_date' => ['sometimes', 'nullable', 'date'],
            'reservation_date' => ['sometimes', 'nullable', 'date'],
            'meal_plan_id' => ['sometimes', 'integer', 'exists:ac_meal_plans,id'],
            'bed_type' => ['sometimes', 'in:king,queen,single,double,sofa'],
            'room_type_id' => ['sometimes', 'integer', 'exists:ac_room_types,id'],
            'room_view_id' => ['sometimes', 'integer', 'exists:ac_room_views,id'],
            'extra_bed' => ['sometimes', 'integer', 'min:0'],
            'currency_id' => ['sometimes', $financial, 'integer', 'exists:currencies,id'],
            'adult_price' => ['sometimes', $financial, 'numeric', 'min:0'],
            'child_price' => ['sometimes', $financial, 'numeric', 'min:0'],
            'extra_bed_price' => ['sometimes', 'numeric', 'min:0'],
            'commission' => ['sometimes', 'numeric', 'min:0'],
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
            'tax_rate_price' => ['sometimes', 'numeric', 'min:0'],
            'discount' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'discount_type' => ['sometimes', 'nullable', 'in:value,percentage'],
        ];

        $validator = Validator::make($request, $rules);
        $validator->after(function ($v) use ($request) {
            if (($request['discount_type'] ?? null) === 'percentage' && ($request['discount'] ?? null) !== null) {
                if ((float) $request['discount'] > 100) {
                    $v->errors()->add('discount', 'When discount_type is percentage, discount must be <= 100');
                }
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $room = $this->roomRepository->findByIdOrFail($request['id']);

        // capacity validation against profile considering update delta
        $reservationId = (int) ($request['reservation_id'] ?? $room->reservation_id);
        $profileId = (int) ($request['profile_id'] ?? $room->profile_id);
        $nationalityId = (int) ($request['nationality_id'] ?? $room->nationality_id ?? 0);
        if ($nationalityId > 0) {
            $totals = $this->roomRepository->getTotalsByReservationProfileNationality($reservationId, $profileId, $nationalityId, $room->id);
            $currentAdults = (int) ($totals->adults ?? 0);
            $currentChildren = (int) ($totals->children ?? 0);
            $currentNoFee = 0;
        } else {
            $totals = $this->roomRepository->getTotalsByReservationProfile($reservationId, $profileId, $room->id);
            $currentAdults = (int) ($totals->adults ?? 0);
            $currentChildren = (int) ($totals->children ?? 0);
            $currentNoFee = (int) ($totals->no_fee_children ?? 0);
        }

        $newAdults = (int) ($request['adult_count'] ?? $room->adult_count);
        $newChildren = (int) ($request['childreen_count'] ?? $room->childreen_count);
        $newNoFee = (int) ($request['no_fee_children_count'] ?? $room->no_fee_children_count);

        $profile = (new \App\sys\Repository\Profile\ProfileRepository)->findByIdOrFail($profileId);
        if ($nationalityId > 0) {
            $travelerTotals = \App\Models\Profile\ProfileTraveler::query()
                ->where('profile_id', $profileId)
                ->where('nationality_id', $nationalityId)
                ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
                ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
                ->first();
            $profileAdults = (int) ($travelerTotals->adults_total ?? 0);
            $profileChildren = (int) ($travelerTotals->children_total ?? 0);
            $profileNoFee = PHP_INT_MAX;
        } else {
            $profileAdults = (int) $profile->adults_count;
            $profileChildren = (int) $profile->children_count;
            $profileNoFee = (int) $profile->children_no_fee_count;
        }

        if ($currentAdults + $newAdults > $profileAdults) {
            $this->setError(['adult_count' => ['Exceeds profile adults_count']]);

            return false;
        }
        if ($currentChildren + $newChildren > $profileChildren) {
            $this->setError(['childreen_count' => ['Exceeds profile children_count']]);

            return false;
        }
        if ($currentNoFee + $newNoFee > $profileNoFee) {
            $this->setError(['no_fee_children_count' => ['Exceeds profile children_no_fee_count']]);

            return false;
        }

        // Map optional reservation_date to booking_date if provided
        if (! empty($request['reservation_date']) && empty($request['booking_date'])) {
            $request['booking_date'] = $request['reservation_date'];
        }

        // recompute if dates or price inputs changed
        $payload = $request;
        $checkIn = $payload['check_in'] ?? $room->check_in;
        $checkOut = $payload['check_out'] ?? $room->check_out;
        $nights = $this->computeNights($checkIn, $checkOut);
        $payload['nights_count'] = $nights;

        $financials = $this->computeFinancials(array_merge([
            'adult_count' => $payload['adult_count'] ?? $room->adult_count,
            'childreen_count' => $payload['childreen_count'] ?? $room->childreen_count,
            'extra_bed' => $payload['extra_bed'] ?? $room->extra_bed,
            'adult_price' => $payload['adult_price'] ?? $room->adult_price,
            'child_price' => $payload['child_price'] ?? $room->child_price,
            'extra_bed_price' => $payload['extra_bed_price'] ?? $room->extra_bed_price,
            'commission' => $payload['commission'] ?? $room->commission,
            'tax_rate_price' => $payload['tax_rate_price'] ?? $room->tax_rate_price,
            'discount' => $payload['discount'] ?? $room->discount,
            'discount_type' => $payload['discount_type'] ?? $room->discount_type,
        ], [
            'check_in' => $checkIn,
            'check_out' => $checkOut,
        ]), $nights);

        $payload = array_merge($payload, $financials);

        $beforeTotal = (float) $room->total_amount;
        $beforeTaxPerNight = (float) ($room->tax_rate_price ?? 0);
        $beforeNights = (int) $room->nights_count;
        $canFinancial = Gate::allows('EDIT_ROOM_COST');

        $updated = $this->roomRepository->update($room, $payload);

        if ($canFinancial) {
            // adjust aggregated price row
            $reservationId = (int) ($payload['reservation_id'] ?? $room->reservation_id);
            $profileId = (int) ($payload['profile_id'] ?? $room->profile_id);
            $currencyId = (int) ($payload['currency_id'] ?? $room->currency_id);
            $afterTotal = (float) $updated->total_amount;
            $afterTaxPerNight = (float) ($updated->tax_rate_price ?? 0);
            $afterNights = (int) $updated->nights_count;

            // Recalculate taxes for this room
            $finalCurrencyId = $currencyId;
            $currency = Currency::find((int) $finalCurrencyId);
            $exchangeRate = (float) ($currency->exchange_rate ?? 1);
            // Determine tax ids to use
            $incomingTaxIds = $payload['tax_rate_id'] ?? null;
            if (is_null($incomingTaxIds)) {
                // fallback to existing linked taxes
                $existingTaxRows = AccommodationReservationRoomTax::where('accommodation_reservation_rooms_id', (int) $room->id)->pluck('tax_rate_id')->toArray();
                $incomingTaxIds = $existingTaxRows;
            }
            if (! is_array($incomingTaxIds)) {
                $incomingTaxIds = [$incomingTaxIds];
            }
            $taxRates = empty($incomingTaxIds) ? collect() : TaxRate::whereIn('id', $incomingTaxIds)->get();
            // reset and re-insert
            AccommodationReservationRoomTax::where('accommodation_reservation_rooms_id', (int) $room->id)->delete();
            $totalTaxAmount = 0.0;
            foreach ($taxRates as $taxRate) {
                $calculatedAmount = 0.0;
                if (strtolower((string) $taxRate->type) === 'general') {
                    $percentage = (float) $taxRate->percentage;
                    $calculatedAmount = $afterTotal * ($percentage / 100.0);
                } elseif (strtolower((string) $taxRate->type) === 'revenue') {
                    $calculatedAmount = 0.0;
                }
                AccommodationReservationRoomTax::create([
                    'tax_rate_id' => (int) $taxRate->id,
                    'tax_amount' => $calculatedAmount,
                    'tax_rate_amount' => $exchangeRate,
                    'currency_id' => (int) $finalCurrencyId,
                    'accommodation_reservation_rooms_id' => (int) $room->id,
                    'accommodation_reservation_id' => $reservationId,
                    'profile_id' => $profileId,
                ]);
                $totalTaxAmount += $calculatedAmount;
            }

            // Persist per-night tax on the room row after update
            $taxPerNightAfter = $afterNights > 0 ? ($totalTaxAmount / $afterNights) : 0.0;
            $updated->tax_rate_price = $taxPerNightAfter;
            $updated->save();

            // upsert aggregated invoice service by currency
            $rese = AccommodationReservation::find($reservationId);
            $invoice = $this->upsertInvoiceForReservationCurrency(
                $reservationId,
                $profileId,
                $currencyId,
                $rese?->supplier_id,
                (int) $updated->id,
                $afterTotal,
                $totalTaxAmount,
                $taxRates->mapWithKeys(fn ($t) => [(int) $t->id => (float) ($t->type === 'general' ? $afterTotal * ((float) $t->percentage / 100.0) : 0)])->toArray(),
                $updated->invoice_services_id
            );
            if ($invoice) {
                $updated->invoice_services_id = $invoice->id;
                $updated->save();
            }
        }

        return $updated;
    }

    private function authorizeBasic($mode)
    {
        $ability = $mode == 'edit' ? 'EDIT_ROOM_DETAILS' : 'ADD_ROOM_DETAILS';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    private function authorizeFinancial($mode)
    {
        $ability = $mode == 'edit' ? 'EDIT_ROOM_COST' : 'ADD_ROOM_COST';

        return Gate::allows($ability) ? 'required' : 'nullable';
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_accommodation_reservation_rooms,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // we need to decrease aggregated totals before deleting
        $rooms = $this->roomRepository->del($ids);
        foreach ($rooms as $room) {
            $existing = $this->priceRepository->findByReservationProfileCurrency((int) $room->reservation_id, (int) $room->profile_id, (int) $room->currency_id);
            if ($existing) {
                $amountDelta = -1 * (float) $room->total_amount;
                $taxDelta = -1 * ((float) ($room->tax_rate_price ?? 0) * (int) $room->nights_count);
                $this->priceRepository->incrementTotals($existing, $amountDelta, $taxDelta);
            }
        }

        return $this->roomRepository->deleteByIds($ids);
    }

    public function getColumnChanges($id, $columnName)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation_rooms,id'],
            'columnName' => ['required', 'string', 'max:255'],
        ];
        $validator = Validator::make(['id' => $id, 'columnName' => $columnName], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $room = $this->getById($id);
        if (! $room) {
            return false;
        }

        return $this->roomRepository->getColumnChanges($room, $columnName);
    }
}
