<?php

namespace App\Http\Controllers\Accommodation;

use App\Http\Controllers\Controller;
use App\Http\Resources\Accommodation\AccommodationHistoryResource;
use App\Http\Resources\AccommodationsCollection;
use App\Http\Resources\AccommodationsResource;
use App\sys\ApiResponse;
use App\sys\Services\Accommodation\AccommodationsService;
use Illuminate\Http\Request;

class AccommodationsController extends Controller
{
    use ApiResponse;

    protected AccommodationsService $accommodationsService;

    private $accommodationsTranslationService;

    public function __construct(AccommodationsService $accommodationsService)
    {
        $this->accommodationsService = $accommodationsService;
        $this->accommodationsTranslationService = new \App\sys\Services\Accommodation\AccommodationsTranslationService;
    }

    public function index()
    {
        $items = $this->accommodationsService->getPaginatedAccommodations();

        return $this->apiResponse(200, 'success', null, new AccommodationsCollection($items));
    }

    public function show(int $id)
    {
        $item = $this->accommodationsService->getAccommodationById($id);
        if (! $item) {
            $errors = $this->accommodationsService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationsResource($item));
    }

    public function store(Request $request)
    {
        $data = $request->all();
        $response = $this->accommodationsService->createAccommodation($data);
        if (! $response) {
            $errors = $this->accommodationsService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationsResource($response));
    }

    public function update(Request $request)
    {
        $data = $request->all();
        $response = $this->accommodationsService->updateAccommodation($data);
        if (! $response) {
            $errors = $this->accommodationsService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationsResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->accommodationsService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Accommodations deleted successfully', null, null);
        }
        $errors = $this->accommodationsService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getActive()
    {
        $data = $this->accommodationsService->getActive();
        $collect = new \App\Http\Resources\AccommodationsCollection($data);

        return $this->apiResponse(200, 'Active Accommodations retrieved successfully', null, $collect);
    }

    public function getByType(string $type, $city_id)
    {
        if ($accommodations = $this->accommodationsService->getByType($type, $city_id)) {
            return $this->apiResponse(200, 'success', null, new AccommodationsCollection($accommodations));
        }
        $errors = $this->accommodationsService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->accommodationsTranslationService->save($request->all(), $id)) {
            $errors = $this->accommodationsTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        $data = $this->accommodationsService->getByIdWithTranslation($id);
        if ($data) {
            $collection = new \App\Http\Resources\Accommodation\AccommodationsLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->accommodationsService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->accommodationsTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->accommodationsTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $accommodation = $this->accommodationsService->getAccommodationById($id);
        if (! $accommodation) {
            $errors = $this->accommodationsService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->accommodationsService->getAllChanges($accommodation);

        return $this->apiResponse(200, 'success', null, AccommodationHistoryResource::collection($changes));
    }
}
