<?php

namespace App\Http\Controllers\Accounting\Reports;

use App\Http\Controllers\Controller;
use App\Models\Accounting\CostCenter;
use App\Models\General\Companies;
use App\Models\Languages\Languages;
use App\sys\ApiResponse;
use App\sys\Services\Accounting\ReportService;
use Illuminate\Http\Request;
use PDF;

class ReportController extends Controller
{
    use ApiResponse;

    protected ReportService $service;

    public function __construct(ReportService $service)
    {
        $this->service = $service;
    }
public function dailyReport(Request $request)
{
    $reports = $this->service->getReports();

    if ($request->get('view') === 'download') {
        $companyIds = json_decode($request->query('company_ids', '[]'), true);
        $companies = $this->getCompanies($companyIds);
        $type = 'Daily ';
        $direction = config('app.direction', 'ltr');
        $ShowByDefault = $request->query('currency_id') ? false : true;

        $pdf = PDF::loadView('pdf.reports.daily_report', [
            'data' => $reports,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'type' => $type,
            'companies' => $companies,
            'direction' => $direction,
            'ShowByDefault' => $ShowByDefault
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Daily_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, $reports);
}

public function balanceAndIncome(Request $request)
{
    $reports = $this->service->getHierarchicalTreeAccountReport();

    if (!$reports) {
        $errors = $this->service->errors();
        return $this->apiResponse(400, 'fail', $errors, null);
    }

    if ($request->query('view') === 'download') {
        $companyIds = json_decode($request->query('company_ids', '[]'), true);
        $companies = $this->getCompanies($companyIds);
        $direction = config('app.direction', 'ltr');
        $reportData = $reports['data']['data'] ?? $reports['data'] ?? $reports;
        $totals = $reports['data']['totals'] ?? $reports['totals'] ?? ['grand_total_balance' => 0];

        $pdf = PDF::loadView('pdf.reports.balance-income', [
            'data' => $reportData,
            'totals' => $totals,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'type' => $request->get('type'),
            'companies' => $companies,
            'direction' => $direction
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Balance_and_Income_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, ['reports' => $reports, 'companies' => $companies]);
}

public function trailBalance(Request $request)
{
    $reports = $this->service->getHierarchicalTreeAccountTrailBalance();

    if (!$reports) {
        $errors = $this->service->errors();
        return $this->apiResponse(400, 'fail', $errors, null);
    }

    if ($request->query('view') === 'download') {
        $companyIds = json_decode($request->query('company_ids', '[]'), true);
        $companies = $this->getCompanies($companyIds);
        $type = 'General Ledger';
        $direction = config('app.direction', 'ltr');

        $pdf = PDF::loadView('pdf.reports.trial_balance', [
            'data' => $reports,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'type' => $type,
            'companies' => $companies,
            'direction' => $direction
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Trial_Balance_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, $reports);
}

public function accountTransfers(Request $request)
{
    $reports = $this->service->treeAccountTransfers($request->all());

    if (!$reports) {
        $errors = $this->service->errors();
        return $this->apiResponse(400, 'fail', $errors, null);
    }

    if ($request->query('view') === 'download') {
        $direction = config('app.direction', 'ltr');
        $reportData = $reports['data'] ?? $reports;
        $ShowByDefault = $request->input('show_by_default');
        $companyIds = $request->input('company_ids', []);
        $companies = $this->getCompanies($companyIds);
        $type = 'Bank Statement';

        $pdf = PDF::loadView('pdf.reports.Bank-statement', [
            'data' => $reportData,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'companies' => $companies,
            'direction' => $direction,
            'type' => $type,
            'ShowByDefault' => $ShowByDefault,
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Bank_Statement_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, $reports);
}

public function ledgerReport(Request $request)
{
    $reports = $this->service->treeAccountTransfersWithInitialBalance($request->all());

    if (!$reports) {
        $errors = $this->service->errors();
        return $this->apiResponse(400, 'fail', $errors, null);
    }

    if ($request->query('view') === 'download') {
        $direction = config('app.direction', 'ltr');
        $reportData = $reports['data'] ?? $reports;
        $companyIds = $request->input('company_ids', []);
        $companies = $this->getCompanies($companyIds);
        $ShowByDefault = $request->input('show_by_default');
        $type = 'Ledger';

        $pdf = PDF::loadView('pdf.reports.Ledger', [
            'data' => $reportData,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'companies' => $companies,
            'direction' => $direction,
            'type' => $type,
            'ShowByDefault' => $ShowByDefault,
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Ledger_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, $reports);
}

public function costCenterReport(Request $request)
{
    $reports = $this->service->costCenterReport($request->all());

    if (!$reports) {
        $errors = $this->service->errors();
        return $this->apiResponse(400, 'fail', $errors, null);
    }

    if ($request->query('view') === 'download') {
        $companyIds = $request->input('company_ids', []);
        $cost_center_ids = $request->input('cost_center_ids', []);
        $companies = $this->getCompanies($companyIds);
        $costCenters = $this->getCostCenters($cost_center_ids);
        $type = 'Cost Center';
        $direction = config('app.direction', 'ltr');

        $pdf = PDF::loadView('pdf.reports.CostCenter', [
            'data' => $reports,
            'date_from' => $request->get('date_from'),
            'date_to' => $request->get('date_to'),
            'type' => $type,
            'companies' => $companies,
            'costCenters' => $costCenters,
            'direction' => $direction
        ], [], array_merge(
            config('pdf_custom.mpdf'),
            ['directionality' => $direction]
        ));

        return $pdf->download("Cost_Center_Report_" . time() . ".pdf");
    }

    return $this->apiResponse(200, 'success', null, $reports);
}


public function getCompanies( $companyIds)
{

     return !empty($companyIds) ? Companies::whereIn('id', $companyIds)
          ->select('id', 'name_company', 'company_vat')
          ->get()
      : collect(); //
}


public function getCostCenters( $costCenterIds)
{

     return !empty($costCenterIds) ? CostCenter::whereIn('id', $costCenterIds)
          ->select('id', 'name')
          ->get()
      : collect(); //

}


}
