<?php

namespace App\Http\Controllers\Profile;

use App\Http\Controllers\Controller;
use App\Http\Resources\Profile\AccommodationReservationRoomCollection;
use App\Http\Resources\Profile\AccommodationReservationRoomResource;
use App\Http\Resources\Profile\AccommodationRoomColumnHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\Profile\AccommodationReservationRoomService;
use Illuminate\Http\Request;

class AccommodationReservationRoomController extends Controller
{
    use ApiResponse;

    protected AccommodationReservationRoomService $roomService;

    public function __construct(AccommodationReservationRoomService $roomService)
    {
        $this->roomService = $roomService;
    }

    public function index()
    {
        $items = $this->roomService->getAll();

        return $this->apiResponse(200, 'success', null, new AccommodationReservationRoomCollection($items));
    }

    public function show(int $id)
    {
        $item = $this->roomService->getById($id);
        if (! $item) {
            $errors = $this->roomService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, (new AccommodationReservationRoomResource($item))->includeChanges());
    }

    public function store(Request $request)
    {
        $response = $this->roomService->create($request->all());
        if (! $response) {
            $errors = $this->roomService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationRoomResource($response));
    }

    public function update(Request $request)
    {
        $response = $this->roomService->update($request->all());

        if (! $response) {
            $errors = $this->roomService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationRoomResource($response));
    }

    public function destroy(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->roomService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Accommodation reservation rooms deleted successfully', null, null);
        }
        $errors = $this->roomService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllByReservationId(int $id)
    {
        $items = $this->roomService->getAllByReservationId($id);

        if (! $items) {
            $errors = $this->roomService->errors();

            return $this->apiResponse(400, 'fail', null, $errors);
        }

        return $this->apiResponse(200, 'success', null, new AccommodationReservationRoomCollection($items));
    }

    public function getColumChanges($id, $columnName)
    {
        $columnHistory = $this->roomService->getColumnChanges($id, $columnName);

        if (! $columnHistory) {
            $errors = $this->roomService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, AccommodationRoomColumnHistoryResource::collection($columnHistory));
    }
}
