<?php

namespace App\sys;

use App\Models\Accounting\AccountsStructure;
use App\Models\Accounting\TreeAccounting;
use App\Models\General\Currency;
use App\sys\Repository\Accounting\TreeAccountingRepository;

class Helper
{
    public static function saveFiles(string $base64, string $folder = 'uploads', string $type = 'general', array $allowedExtensions = []): array|false
    {
        if (! $base64 || ! str_starts_with($base64, 'data:')) {
            return [
                'status' => false,
                'errors' => ['البيانات غير صالحة أو غير بصيغة base64'],
            ];
        }

        // استخراج النوع والامتداد
        if (! preg_match('/^data:(\w+\/\w+);base64,/', $base64, $matches)) {
            return [
                'status' => false,
                'errors' => ['تعذر التعرف على نوع الملف من base64'],
            ];
        }

        $mimeType = $matches[1]; // مثل: image/png أو application/pdf
        $extension = explode('/', $mimeType)[1]; // png أو pdf أو jpeg ...
        $extension = strtolower($extension);

        // تحديد الامتدادات المسموحة
        $defaultAllowed = match ($type) {
            'image' => ['jpg', 'jpeg', 'png', 'gif', 'webp'],
            'file' => ['pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt'],
            default => ['jpg', 'jpeg', 'png', 'gif', 'webp', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt']
        };

        $finalAllowed = count($allowedExtensions) > 0 ? $allowedExtensions : $defaultAllowed;

        if (! in_array($extension, $finalAllowed)) {
            return [
                'status' => false,
                'errors' => ['امتداد الملف غير مسموح به: '.$extension],
            ];
        }

        // استخراج بيانات الصورة / الملف
        $data = substr($base64, strpos($base64, ',') + 1);
        $decoded = base64_decode($data);

        if ($decoded === false) {
            return [
                'status' => false,
                'errors' => ['فشل في فك تشفير base64'],
            ];
        }

        // إنشاء المسار والاسم
        $folderPath = public_path($folder);
        if (! file_exists($folderPath)) {
            mkdir($folderPath, 0755, true);
        }

        $fileName = uniqid().'.'.$extension;
        $filePath = $folderPath.'/'.$fileName;

        file_put_contents($filePath, $decoded);

        return [
            'status' => true,
            'path' => $folder.'/'.$fileName,
            'extension' => $extension,
            'type' => $type,
        ];
    }

    /**
     * تُستخدم هذه الدالة لإنشاء هيكل مجموع (grouped structure) من الحسابات (TreeAccounting)
     * مجمعة حسب العملة (Currency)، مع إنشاء بيانات حقول الإدخال (input metadata)
     * بشكل ديناميكي بناءً على Enum يتم تمريره.
     *
     * ✅ حالات الاستخدام (Use Case):
     * تُفيدك هذه الدالة لو كنت عايز تبني فورم أو واجهة ديناميكية، بحيث:
     * - كل عملة يكون لها شجرة الحسابات الخاصة بها.
     * - وكل شجرة تحتوي على حقول إدخال (Inputs) مبنية على Enum معيّن زي CompanyAccountMapping.
     *
     * 🔧 طريقة عمل الدالة (How It Works):
     * 1. بتجيب كل الحسابات النشطة من TreeAccountingRepository (ممكن تحديد عملات معينة).
     * 2. بتجمع الحسابات حسب currency_id.
     * 3. لكل مجموعة:
     *    - بتجهز الـ tree (قائمة الحسابات).
     *    - وبتستخدم الـ Enum الممرّر (عن طريق ::cases()) علشان تولد حقول الإدخال.
     *    - كل حقل بياخد اسم (input_name) وقيمة مميزة (label_name) متكونة من قيمة الـ Enum واسم العملة.
     *
     * 🧾 شكل البيانات اللي بترجعها (Example Output):
     * [
     *     [
     *         'inputs' => [
     *             [
     *                 'input_name' => 'main_account',
     *                 'label_name' => 'main_account_USD',
     *                 'currency_id' => 1,
     *                 'currency_name' => 'USD'
     *             ],
     *             ...
     *         ],
     *         'tree' => [
     *             [
     *                 'id' => 5,
     *                 'name' => 'Cash Account',
     *             ],
     *             ...
     *         ]
     *     ],
     *     ...
     * ]
     *
     * @param  string  $enumClass  اسم الـ Enum الكامل (fully qualified class name)،
     *                             مثل: \App\Enums\CompanyAccountMapping::class
     *                             ويشترط يكون فيه method اسمها cases() بتُرجع قائمة بالقيم.
     * @param  array  $currencyIds  (اختياري) مصفوفة من أرقام العملات (IDs) لتصفية النتائج.
     * @return array بترجع مصفوفة فيها مجموعات، كل مجموعة تحتوي على:
     *               - inputs: بيانات الحقول المرتبطة بالعملة
     *               - tree: الحسابات المرتبطة بالعملة
     *
     * @throws \InvalidArgumentException لو الـ Enum اللي اترسل مش صالح أو مش موجود.
     */
    public static function getCurrencyWithtree(string $enumClass, array $currencyIds = []): array
    {
        if (! enum_exists($enumClass)) {
            throw new \InvalidArgumentException("Invalid enum class: $enumClass");
        }

        $accountMappingCases = $enumClass::cases();

        $activeAccounts = TreeAccountingRepository::getTreeByCurrency($currencyIds);
        $groupedAccounts = [];

        foreach ($activeAccounts as $currencyId => $accounts) {
            $tree = [];
            $inputs = [];
            $currencyName = null;

            foreach ($accounts as $account) {
                $tree[] = [
                    'id' => $account->id,
                    'name' => $account->title,
                ];

                $currencyName ??= $account->currency->name;
            }

            foreach ($accountMappingCases as $case) {
                $inputs[] = [
                    'input_name' => $case->value,
                    'label_name' => "{$case->value}_{$currencyName}",
                    'currency_id' => $currencyId,
                    'currency_name' => $currencyName,
                ];
            }

            $groupedAccounts[] = [
                'inputs' => $inputs,
                'tree' => $tree,
            ];
        }

        return $groupedAccounts;
    }

    public static function getTreeByCurrency(array $currencyIds = []): array
    {
        $activeAccounts = TreeAccountingRepository::getTreeByCurrency($currencyIds);
        $groupedAccounts = [];
        foreach ($activeAccounts as $currencyId => $accounts) {
            $tree = [];
            $inputs = [];
            $currencyName = null;
            $inputs[] = [
                'currency_id' => $currencyId,
                'currency_name' => $currencyName,
            ];
            foreach ($accounts as $account) {
                $tree[] = [
                    'id' => $account->id,
                    'name' => $account->title,
                ];

                $currencyName ??= $account->currency->name;
            }
            $groupedAccounts[] = [
                'inputs' => $inputs,
                'tree' => $tree,
            ];
        }

        return $groupedAccounts;
    }

    public static function validateCurrencyInputCombinations($validator, $data, $enumClass, $inputName, $currencyIds = [], $label = 'الحسابات')
    {
        $activeCurrencies = Currency::query()
            ->where('status', 1)
            ->when(! empty($currencyIds), function ($query) use ($currencyIds) {
                $query->whereIn('id', $currencyIds);
            })
            ->pluck('name', 'id')
            ->toArray();
        $cases = $enumClass::cases();
        $expectedCombinations = [];
        foreach ($cases as $case) {
            foreach ($activeCurrencies as $currencyId => $currencyName) {
                $expectedCombinations["{$case->value}_{$currencyId}"] = "{$case->value} + {$currencyName}";
            }
        }

        $foundCombinations = [];
        foreach ($data[$inputName] ?? [] as $item) {
            if (! isset($item['input_name'], $item['currency_id'])) {
                continue;
            }

            $key = "{$item['input_name']}_{$item['currency_id']}";
            $foundCombinations[] = $key;
        }

        $missingCombinations = array_diff(array_keys($expectedCombinations), $foundCombinations);

        if (! empty($missingCombinations)) {
            $messages = array_map(
                fn ($key) => "الحقل المطلوب: {$expectedCombinations[$key]}",
                $missingCombinations
            );

            $validator->errors()->add($inputName, "يوجد {$label} ناقصة لكل العملات: ".implode('، ', $messages));

            return false;
        }

        return true;
    }

    public static function getChangedColumns($model)
    {
        if (! $model) {
            return null;
        }

        $allAudits = $model->audits()->get();

        // Collect all unique column names that were changed
        $changedColumns = collect();

        foreach ($allAudits as $audit) {
            // Get columns from old_values
            if (! empty($audit->old_values)) {
                $changedColumns = $changedColumns->merge(array_keys($audit->old_values));
            }

            // Get columns from new_values
            if (! empty($audit->new_values)) {
                $changedColumns = $changedColumns->merge(array_keys($audit->new_values));
            }
        }

        // Remove duplicates and sort
        $uniqueChangedColumns = $changedColumns->unique()->sort()->values();

        return $uniqueChangedColumns;
    }

    public static function getColumnHistory($model, $columnName)
    {
        return $model->audits()
            ->where(function ($query) use ($columnName) {
                $query->whereJsonContains('old_values->'.$columnName, '*')
                    ->orWhereJsonContains('new_values->'.$columnName, '*');
            })
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($audit) use ($columnName) {
                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->name : null,
                    'old_value' => $audit->old_values[$columnName] ?? null,
                    'new_value' => $audit->new_values[$columnName] ?? null,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            });
    }

    /**
public  static function  generateNextSerial($parentId = null, $levelId) {
        // 1. تعريف المستوى الحالي
        $structure = AccountsStructure::where('id', $levelId)->first();
        if (!$structure) {
            throw new \Exception("Structure for level {$levelId} not found");
        }

        $initial = $structure->start_position;   // بداية الخانة
        $final   = $structure->end_position;     // نهاية الخانة
        $length  = $structure->field_length;

        // 2. لو مفيش parent (المستوى الأول)
        if (!$parentId) {
            $last =TreeAccounting::where('accounts_structure_id', $levelId)
                ->orderBy('serial_number', 'desc')
                ->first();

            $next = $last ? intval(substr($last->serial_number, $initial-1, $length)) + 1 : 1;
            $nextPadded = str_pad($next, $length, '0', STR_PAD_LEFT);

            // اعمل الكود كامل: الباقي كله صفر
            $code = str_repeat("0", $final);
            $code = substr_replace($code, $nextPadded, $initial-1, $length);

            // حط داشات للتقسيم زي الشكل بتاعك
            return self::formatSerial($code, $levelId);
        }

        // 3. لو فيه parent
        $parent = TreeAccounting::where('id', $parentId)->first();
        if (!$parent) {
            throw new \Exception("Parent not found");
        }

        // 4. هات آخر رقم مستخدم تحت نفس الأب
        $last = TreeAccounting::where('accounts_structure_id', $levelId)
            ->where('tree_accounting_id', $parentId)
            ->orderBy('serial_number', 'desc')
            ->first();

        $next = $last ? intval(substr($last->serial_number, $initial-1, $length)) + 1 : 1;
        $nextPadded = str_pad($next, $length, '0', STR_PAD_LEFT);

        // 5. ركّب الكود بناءً على كود الأب
        $code = $parent->serial_number;
        $code = substr_replace($code, $nextPadded, $initial-1, $length);

        return self::formatSerial($code, $levelId);
    }

// دالة لتنسيق الكود وإضافة الـ dashes
 public static function formatSerial($code, $levelId) {
        // ممكن هنا نجيب الـ structure كله ونحط الـ dashes بين الحقول
        // مثال ثابت: 1-1-01-00-000
        return substr($code,0,1).'-'.substr($code,1,1).'-'.substr($code,2,2).'-'.substr($code,4,2).'-'.substr($code,6,3);
    }

     **/
    public static function generateNextSerial($parentId = null, $levelId = null)
    {
        // 1) تحديد المستوى المطلوب بناءً على الأب إن لم يُمرَّر صراحةً
        if ($levelId === null) {
            if (! $parentId || $parentId == 0) {
                // المستوى الأول افتراضياً هو id=1
                $levelId = 1;
            } else {
                $parent = TreeAccounting::where('id', $parentId)->first();
                if (! $parent) {
                    return [
                        'status' => false,
                        'message' => 'Parent account not found',
                        'errors' => ['Parent account not found'],
                    ];
                }
                // المستوى التالي بعد مستوى الأب
                $levelId = ((int) $parent->accounts_structure_id) + 1;
            }
        }

        // 2) تحميل هيكل المستوى الحالي والتأكد من وجوده
        $structure = AccountsStructure::where('id', $levelId)->first();
        if (! $structure) {
            return [
                'status' => false,
                'message' => "Accounts structure level not found: {$levelId}",
                'errors' => ["Accounts structure level not found: {$levelId}"],
            ];
        }

        $initial = $structure->start_position;   // بداية الخانة
        $final = $structure->end_position;     // نهاية الخانة
        $length = $structure->field_length;

        // 3) لو مفيش parent (المستوى الأول)
        if (! $parentId) {
            $last = TreeAccounting::where('accounts_structure_id', $levelId)
                ->orderBy('serial_number', 'desc')
                ->first();

            // احسب الجزء من الرقم (بدون شرطات) بعد عمل padding لطول كامل
            $lastDigits = $last ? str_pad((string) $last->serial_number, $final, '0', STR_PAD_LEFT) : null;
            $lastSegment = $lastDigits ? intval(substr($lastDigits, $initial - 1, $length)) : 0;
            // تحقق من الوصول للحد الأقصى للمستوى الحالي بناءً على field_length
            $maxSegment = (int) (pow(10, $length) - 1);
            if ($lastSegment >= $maxSegment) {
                return [
                    'status' => false,
                    'message' => 'لا يمكن إضافة المزيد لهذا المستوى',
                    'errors' => [
                        'max_reached' => [
                            'تم الوصول إلى أقصى رقم مسموح به للمستوى الحالي (field_length='.$length.', max='.$maxSegment.')',
                        ],
                    ],
                ];
            }

            $next = $lastSegment + 1;
            $nextPadded = str_pad($next, $length, '0', STR_PAD_LEFT);

            // اعمل الكود كامل: الباقي كله صفر
            $code = str_repeat('0', $final);
            $code = substr_replace($code, $nextPadded, $initial - 1, $length);

            return [
                'serial_number' => intval($code),
                'serial_number_dight' => self::formatSerial($code, $levelId),
                'start_position' => $initial,
                'end_position' => $final,
                'field_length' => $length,
                'last_segment_value' => $lastSegment,
                'next_segment_value' => $next,
                'accounts_structure_id' => $levelId,
                'parent_id' => $parentId,
                'title' => $structure->title,
                'parent_currency' => null,
            ];
        }

        // 4) لو فيه parent
        $parent = TreeAccounting::where('id', $parentId)->first();
        if (! $parent) {
            return [
                'status' => false,
                'message' => 'Parent account not found',
                'errors' => ['Parent account not found'],
            ];
        }

        // 5) هات آخر رقم مستخدم تحت نفس الأب
        $last = TreeAccounting::where('accounts_structure_id', $levelId)
            ->where('tree_accounting_id', $parentId)
            ->orderBy('serial_number', 'desc')
            ->first();

        $lastDigits = $last ? str_pad((string) $last->serial_number, $final, '0', STR_PAD_LEFT) : null;
        $lastSegment = $lastDigits ? intval(substr($lastDigits, $initial - 1, $length)) : 0;
        // تحقق من الوصول للحد الأقصى للمستوى الحالي بناءً على field_length
        $maxSegment = (int) (pow(10, $length) - 1);
        if ($lastSegment >= $maxSegment) {
            return [
                'status' => false,
                'message' => 'لا يمكن إضافة المزيد لهذا المستوى',
                'errors' => [
                    'max_reached' => [
                        'تم الوصول إلى أقصى رقم مسموح به للمستوى الحالي (field_length='.$length.', max='.$maxSegment.')',
                    ],
                ],
            ];
        }

        $next = $lastSegment + 1;
        $nextPadded = str_pad($next, $length, '0', STR_PAD_LEFT);

        // 6) ★ كوّن كود كامل بطول مستوى الابن، ثم انسخ كود الأب في المقدمة
        $parentLevelId = (int) $parent->accounts_structure_id;
        $parentStruct = AccountsStructure::where('id', $parentLevelId)->first();
        $parentFinal = $parentStruct ? (int) $parentStruct->end_position : ($initial - 1);

        $parentDigits = str_pad((string) $parent->serial_number, $parentFinal, '0', STR_PAD_LEFT);
        // ابني كود بطول مستوى الابن كله أصفار
        $code = str_repeat('0', $final);
        // ضع مقدمة الأب في بداية الكود
        $code = substr_replace($code, $parentDigits, 0, $parentFinal);
        // غيّر خانة المستوى الحالي فقط
        $code = substr_replace($code, $nextPadded, $initial - 1, $length);

        return [
            'serial_number' => intval($code),
            'serial_number_dight' => self::formatSerial($code, $levelId),
            'start_position' => $initial,
            'end_position' => $final,
            'field_length' => $length,
            'last_segment_value' => $lastSegment,
            'next_segment_value' => $next,
            'accounts_structure_id' => $levelId,
            'parent_id' => $parentId,
            'title' => $structure->title,
            'parent_currency' => $parent->currency_id,
        ];
    }

    // دالة لتنسيق الكود وإضافة الـ dashes
    public static function formatSerial($code, $levelId)
    {
        // صياغة ديناميكية حسب هيكل المستويات من جدول accounts_structure
        // مثال: لو الأطوال [1,1,2,2,3] -> 1-1-01-00-000
        $structures = AccountsStructure::orderBy('id')->get(['id', 'field_length']);
        if ($structures->isEmpty()) {
            return (string) $code;
        }

        // استخدم كل المستويات لتعريف التقسيم دائماً حتى مع مستويات لاحقة صفرية
        $lengths = $structures->pluck('field_length')->map(fn ($l) => (int) $l)->values()->all();

        $digits = (string) $code;
        // تأكد من أن الطول كافٍ عبر padding للأصفار من اليسار حتى إجمالي جميع المستويات
        $totalLength = array_sum($lengths);
        if (strlen($digits) < $totalLength) {
            $digits = str_pad($digits, $totalLength, '0', STR_PAD_LEFT);
        }

        $parts = [];
        $offset = 0;
        foreach ($lengths as $len) {
            $parts[] = substr($digits, $offset, $len);
            $offset += $len;
        }

        return implode('-', $parts);
    }

    /**
     * التأكد من أن الجزء الخاص بالمستوى الحالي فريد أسفل نفس الأب
     * $candidateCode: إما كود الأرقام فقط بطول كامل بدون شرطات
     */
    public static function isSerialUnique($parentId, $levelId, string $candidateCode): bool
    {
        $structure = AccountsStructure::where('id', $levelId)->first();
        if (! $structure) {
            throw new \Exception("Structure for level {$levelId} not found");
        }

        $initial = $structure->start_position;
        $length = $structure->field_length;
        $final = $structure->end_position;

        // تأكد من طول الكود (نقوم بعمل pad لليسار بأصفار لو أقصر)
        $candidateCode = str_pad($candidateCode, $final, '0', STR_PAD_LEFT);
        $segment = substr($candidateCode, $initial - 1, $length);

        $query = TreeAccounting::query()
            ->where('accounts_structure_id', $levelId)
            ->when($parentId, fn ($q) => $q->where('tree_accounting_id', $parentId))
            ->when(! $parentId, fn ($q) => $q->where('tree_accounting_id', 0))
            // قارن الجزء الخاص بالمستوى الحالي بعد عمل padding لكامل الرقم
            ->whereRaw(
                'SUBSTRING(LPAD(serial_number, ?, "0"), ?, ?) = ?',
                [$final, $initial, $length, $segment]
            );

        return ! $query->exists();
    }

    public static function SettingsValidations()
    {
        return [
            'default_currency' => 'sometimes|integer|exists:currencies,id',
        ];
    }
}
