<?php

namespace App\sys\Repository\Accommodation;

use App\Models\Accommodation\Rating;

class RatingRepository
{
    private $rating;

    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function __construct()
    {
        $this->rating = new Rating;
    }

    public function add($data)
    {
        $this->rating->name = $data['name'];
        $this->rating->is_active = $data['active'];

        $this->rating->save();

        return $this->rating;
    }

    public function updated($data)
    {
        $rating = $this->rating->find($data['id']);
        $rating->name = $data['name'] ?? $rating->name;
        $rating->is_active = $data['active'] ?? $rating->is_active;
        $rating->save();

        return $rating;
    }

    public function get()
    {
        $name = request('name', null);
        $limit = request('limit', 15);
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');

        return $this->rating->when($name, function ($q, $name) {
            $q->whereHas('translations', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%")->where('lang_id', app('lang_id'));
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('ac_rating_translation', function ($join) {
                    $join->on('ac_rating.id', '=', 'ac_rating_translation.rating_id')
                        ->where('ac_rating_translation.lang_id', app('lang_id'));
                })->orderBy('ac_rating_translation.name', $order)->select('ac_rating.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->orderBy('updated_at', 'desc')->paginate($limit);
    }

    public function getbyId($id)
    {
        return $this->rating->with('currentTranslation')->find($id);
    }

    public function getActive()
    {
        return $this->rating->with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return $this->rating->with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return \App\Models\Accommodation\Rating::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($rating)
    {
        return $rating->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $audit->old_values ?? [],
                    'new_values' => $audit->new_values ?? [],
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
