<?php

namespace App\sys\Services\Accommodation;

use App\sys\Repository\Accommodation\AccommodationsRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class AccommodationsService extends Services
{
    protected AccommodationsRepository $accommodationsRepository;

    public function __construct(AccommodationsRepository $accommodationsRepository)
    {
        $this->accommodationsRepository = $accommodationsRepository;
    }

    public function getPaginatedAccommodations()
    {
        return $this->accommodationsRepository->getPaginated();
    }

    public function getAccommodationById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_accommodations,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->accommodationsRepository->findByIdOrFail($id);
    }

    public function createAccommodation($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'type' => ['required', 'in:hotel,cruise'],
            'rating_id' => ['required', 'integer'],
            'country_id' => ['required', 'integer'],
            'city_id' => ['required', 'integer'],
            'active' => ['required', 'in:0,1'],
            'address' => ['nullable', 'string'],
            'phone' => ['nullable', 'string'],
            'email' => ['nullable', 'string'],
            'is_main_provider' => ['required', 'boolean'],
            'website' => ['nullable', 'string'],
            'taxable' => ['required', 'boolean'],
            'supplier_ids' => ['nullable', 'array'],
            'supplier_ids.*' => ['integer', 'exists:su_supplier,id'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $accommodations = $this->accommodationsRepository->create($request);
        $this->syncTranslations($accommodations);

        return $accommodations;
    }

    public function updateAccommodation($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_accommodations,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'description' => ['sometimes', 'string'],
            'type' => ['sometimes', 'in:hotel,cruise'],
            'rating_id' => ['sometimes', 'integer'],
            'country_id' => ['sometimes', 'integer'],
            'city_id' => ['sometimes', 'integer'],
            'active' => ['sometimes', 'in:0,1'],
            'address' => ['sometimes', 'string'],
            'phone' => ['sometimes', 'string'],
            'email' => ['sometimes', 'string'],
            'is_main_provider' => ['sometimes', 'boolean'],
            'website' => ['sometimes', 'string'],
            'taxable' => ['sometimes', 'boolean'],
            'supplier_ids' => ['nullable', 'array'],
            'supplier_ids.*' => ['integer', 'exists:su_supplier,id'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->accommodationsRepository->findByIdOrFail($request['id']);

        return $this->accommodationsRepository->update($item, $request);
    }

    public function deleteAccommodation(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_accommodations,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->accommodationsRepository->findByIdOrFail($id);

        return $this->accommodationsRepository->delete($item);
    }

    public function del(array $ids)
    {
        return $this->accommodationsRepository->del($ids);
    }

    public function getActive()
    {
        return $this->accommodationsRepository->getActive();
    }

    public function getByType(string $type, $city_id)
    {
        $rules = [
            'type' => ['required', 'in:hotel,cruise'],
            'city_id' => ['required', 'integer', 'exists:cities,id'],
        ];

        $validator = Validator::make(['type' => $type, 'city_id' => $city_id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->accommodationsRepository->getByType($type, $city_id);
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_accommodations,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $accommodations = $this->accommodationsRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($accommodations, true);
    }

    public function getAllChanges($accommodation)
    {
        return $this->accommodationsRepository->getAllChanges($accommodation);
    }
}
