<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\PaytypeTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class PaytypeTranslationService extends Services
{
    private PaytypeTranslationRepository $paytypeTranslationRepository;

    public function __construct()
    {
        $this->paytypeTranslationRepository = new PaytypeTranslationRepository;
    }

    public function save(array $data, int $paytypeId)
    {
        $rules = ['pay_type_id' => 'required|integer|exists:pay_type,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:pay_type_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('pay_type_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('pay_type_id', $paytypeId))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';
            if ($languageId) {
                $key = $paytypeId.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language duplicated for the same Paytype.';
                } else {
                    $combinations[] = $key;
                }
            }
        }

        $validator = Validator::make(array_merge($data, ['pay_type_id' => $paytypeId]), $rules);

        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->paytypeTranslationRepository->save($data, $paytypeId);
    }

    public function get(int $paytypeId)
    {
        return $this->paytypeTranslationRepository->get($paytypeId);
    }

    public function del($id)
    {
        return $this->paytypeTranslationRepository->del($id);
    }
}
