<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\NationalityRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class NationalityService extends Services
{
    protected NationalityRepository $nationalityRepository;

    public function __construct(NationalityRepository $nationalityRepository)
    {
        $this->nationalityRepository = $nationalityRepository;
    }

    public function getPaginatedNationalities()
    {
        return $this->nationalityRepository->getPaginated();
    }

    public function getNationalityById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:nationalities,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->nationalityRepository->findByIdOrFail($id);
    }

    public function createNationality($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $nationality = $this->nationalityRepository->create($request);
        $this->syncTranslations($nationality);

        return $nationality;
    }

    public function updateNationality($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:nationalities,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);
        $nationality = $this->nationalityRepository->findByIdOrFail($request['id']);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->nationalityRepository->update($nationality, $request);
    }

    public function getActive()
    {
        return $this->nationalityRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:nationalities,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $nationality = $this->nationalityRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($nationality, true);
    }

    public function del(array $ids)
    {
        return $this->nationalityRepository->del($ids);
    }

    public function getAllChanges($nationality)
    {
        return $this->nationalityRepository->getAllChanges($nationality);
    }
}
