<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\RestaurantRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class RestaurantService extends Services
{
    protected RestaurantRepository $restaurantRepository;

    public function __construct(RestaurantRepository $restaurantRepository)
    {
        $this->restaurantRepository = $restaurantRepository;
    }

    public function getPaginatedRestaurant()
    {
        return $this->restaurantRepository->getPaginated();
    }

    public function getRestaurantById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_restaurant,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->restaurantRepository->findByIdOrFail($id);
    }

    public function createRestaurant($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'active' => ['required', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $restaurant = $this->restaurantRepository->create($request);
        $this->syncTranslations($restaurant);

        return $restaurant;
    }

    public function updateRestaurant($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_restaurant,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'active' => ['sometimes', 'in:0,1'],
        ];
        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->restaurantRepository->findByIdOrFail($request['id']);

        return $this->restaurantRepository->update($item, $request);
    }

    public function deleteRestaurant(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_restaurant,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->restaurantRepository->findByIdOrFail($id);

        return $this->restaurantRepository->delete($item);
    }

    public function del(array $ids)
    {
        return $this->restaurantRepository->del($ids);
    }

    public function getActive()
    {
        return $this->restaurantRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ge_restaurant,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $restaurant = $this->restaurantRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($restaurant, true);
    }

    public function getAllChanges($restaurant)
    {
        return $this->restaurantRepository->getAllChanges($restaurant);
    }
}
