<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\sys\Enums\ServicesType;

class AdditionalServicesHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {

        return [
            'travel_tourism_type' => ['required', 'in:'.ServicesType::ADDITIONAL_SERVICES->value],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'nationality_id' => ['nullable', 'integer', 'exists:nationalities,id'],
            'people_count' => ['required', 'integer', 'min:0'],
            'daily_cost' => ['required', 'numeric', 'min:0'],
            'execution_date' => ['required', 'date'],
            'supplier_id' => ['nullable', 'integer', 'exists:su_supplier,id'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],

            // computed/prohibited
            // 'currency_rate' => 'prohibited',
            // 'grand_total' => 'prohibited',
        ];
    }

    public function getUpdateRules(array $request): array
    {
        return [
            'currency_id' => ['sometimes', 'required', 'integer', 'exists:currencies,id'],
            'service_id' => ['sometimes', 'required', 'integer', 'exists:services,id'],
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'nationality_id' => ['sometimes', 'integer', 'exists:nationalities,id'],
            'people_count' => ['sometimes', 'required', 'integer', 'min:0'],
            'daily_cost' => ['sometimes', 'required', 'numeric', 'min:0'],
            'execution_date' => ['sometimes', 'required', 'date'],
            'supplier_id' => ['sometimes', 'nullable', 'integer', 'exists:su_supplier,id'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
        ];
    }

    public function prepareForCreate(array $request): array|false
    {
        // Service type optional validation if needed
        $service = GeneralService::find($request['service_id']);
        if ($service && $service->type !== ServicesType::ADDITIONAL_SERVICES->value) {
            return ['errors' => ['service_id' => ['service type must be additional_services']]];
        }

        $currency = Currency::find($request['currency_id']);
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        $people = (int) ($request['people_count'] ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $request['grand_total'] = $people * $dailyCost;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if ($service && $service->type !== ServicesType::ADDITIONAL_SERVICES->value) {
                return ['errors' => ['service_id' => ['service type must be additional_services']]];
            }
        }

        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        $people = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $request['grand_total'] = $people * $dailyCost;

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // no tax rows for additional_services per current requirements
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // no-op
    }
}
