<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\Profile\ProfileTraveler;
use App\sys\Enums\ServicesType;

class AttractionsTemplesHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        return [
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'nationality_id' => ['required', 'integer', 'exists:nationalities,id'],
            'adults_count' => ['required', 'integer', 'min:0'],
            'children_count' => ['required', 'integer', 'min:0'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            // currency_rate computed internally
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ];
    }

    public function getUpdateRules(array $request): array
    {
        $rules = [
            'service_id' => ['sometimes', 'required', 'integer', 'exists:services,id'],
            'nationality_id' => ['sometimes', 'required', 'integer', 'exists:nationalities,id'],
            'adults_count' => ['sometimes', 'required', 'integer', 'min:0'],
            'children_count' => ['sometimes', 'required', 'integer', 'min:0'],
            'currency_id' => ['sometimes', 'required', 'integer', 'exists:currencies,id'],
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
        ];

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        // Validate service type
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::ATTRACTIONS_TEMPLES->value) {
            return ['errors' => ['service_id' => ['service type must be attractions_temples']]];
        }

        // Validate traveler counts by nationality under profile
        $countsOk = $this->validateTravelerCounts(
            (int) $request['profile_id'],
            (int) $request['nationality_id'],
            (int) $request['adults_count'],
            (int) $request['children_count']
        );
        if ($countsOk !== true) {
            return ['errors' => $countsOk]; // wrap errors
        }

        // Set currency rate
        $currency = Currency::find($request['currency_id']);
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Compute totals
        $adultTotal = ((float) ($request['adult_price'] ?? 0)) * (int) $request['adults_count'];
        $childTotal = ((float) ($request['child_price'] ?? 0)) * (int) $request['children_count'];
        $request['adult_total'] = $adultTotal;
        $request['child_total'] = $childTotal;
        $request['grand_total'] = $adultTotal + $childTotal;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        // If service_id provided or different, ensure it's correct type
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::ATTRACTIONS_TEMPLES->value) {
                return ['errors' => ['service_id' => ['service type must be attractions_temples']]];
            }
        }

        // Determine values for validation
        $profileId = (int) ($request['profile_id'] ?? $existing->profile_id);
        $nationalityId = (int) ($request['nationality_id'] ?? $existing->nationality_id);
        $adultsCount = (int) ($request['adults_count'] ?? $existing->adults_count ?? 0);
        $childrenCount = (int) ($request['children_count'] ?? $existing->children_count ?? 0);

        if ($nationalityId) {
            $countsOk = $this->validateTravelerCounts($profileId, $nationalityId, $adultsCount, $childrenCount);
            if ($countsOk !== true) {
                return ['errors' => $countsOk];
            }
        }

        // Refresh currency rate if currency_id provided
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        // Recompute totals
        $adultPrice = (float) ($request['adult_price'] ?? $existing->adult_price ?? 0);
        $childPrice = (float) ($request['child_price'] ?? $existing->child_price ?? 0);
        $adultTotal = $adultPrice * $adultsCount;
        $childTotal = $childPrice * $childrenCount;
        $request['adult_total'] = $adultTotal;
        $request['child_total'] = $childTotal;
        $request['grand_total'] = $adultTotal + $childTotal;

        return $request;
    }

    private function validateTravelerCounts(int $profileId, int $nationalityId, int $adultsRequested, int $childrenRequested)
    {
        $totals = ProfileTraveler::query()
            ->where('profile_id', $profileId)
            ->where('nationality_id', $nationalityId)
            ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
            ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
            ->first();

        if (! $totals || ((int) $totals->adults_total + (int) $totals->children_total) === 0) {
            return [
                'nationality_id' => ['لا يوجد مسافرون بهذه الجنسية مرتبطون بالبروفايل'],
            ];
        }

        $errors = [];
        if ($adultsRequested > (int) $totals->adults_total) {
            $errors['adults_count'][] = 'عدد الكبار المطلوب أكبر من المسجل في المسافرين';
        }
        if ($childrenRequested > (int) $totals->children_total) {
            $errors['children_count'][] = 'عدد الأطفال المطلوب أكبر من المسجل في المسافرين';
        }

        return empty($errors) ? true : $errors;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        // This type currently has no tax persistence here
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        // This type currently has no tax persistence here
    }
}
