<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\TaxRate;
use App\sys\Enums\ServicesType;

class SingleTransportationHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        return [
            'travel_tourism_type' => ['required', 'in:'.ServicesType::SINGLE_TRANSPORTATION->value],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'supplier_id' => ['required', 'integer', 'exists:su_supplier,id'],
            'city_id' => ['required', 'integer', 'exists:cities,id'],
            'city_to_id' => ['required', 'integer', 'exists:cities,id'],
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'people_count' => ['required', 'integer', 'min:0'],
            'daily_cost' => ['required', 'numeric', 'min:0'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'operation_data_id' => ['required', 'integer', 'exists:ge_operation_data,id'],

            'extra_cost' => ['nullable', 'numeric', 'min:0'],
            'executive_id' => ['nullable', 'integer', 'exists:users,id'],
            'notes' => ['nullable', 'string', 'max:1000'],

            // computed/prohibited
            // 'currency_rate' => 'prohibited',
            //  'total_tax' => 'prohibited',
            // 'grand_total' => 'prohibited',

            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];
    }

    public function getUpdateRules(array $request): array
    {
        return [
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'supplier_id' => ['sometimes', 'required', 'integer', 'exists:su_supplier,id'],
            'city_id' => ['sometimes', 'required', 'integer', 'exists:cities,id'],
            'city_to_id' => ['sometimes', 'required', 'integer', 'exists:cities,id'],
            'service_id' => ['sometimes', 'required', 'integer', 'exists:services,id'],
            'people_count' => ['sometimes', 'required', 'integer', 'min:0'],
            'daily_cost' => ['sometimes', 'required', 'numeric', 'min:0'],
            'currency_id' => ['sometimes', 'required', 'integer', 'exists:currencies,id'],
            'operation_data_id' => ['sometimes', 'required', 'integer', 'exists:ge_operation_data,id'],
            'extra_cost' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'executive_id' => ['sometimes', 'nullable', 'integer', 'exists:users,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'invoice_id' => ['sometimes', 'nullable', 'integer'],
            'reserve_number' => ['sometimes', 'nullable', 'string', 'max:255'],
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];
    }

    public function prepareForCreate(array $request): array|false
    {
        $currency = Currency::find($request['currency_id']);
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        $people = (int) ($request['people_count'] ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $extra = (float) ($request['extra_cost'] ?? 0);

        $subtotal = $dailyCost;
        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (is_array($taxRateIds) && ! empty($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        }

        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $extra + $totalTax;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        $people = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $extra = (float) ($request['extra_cost'] ?? $existing->extra_cost ?? 0);

        $subtotal = $dailyCost;
        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? null;
        if (is_array($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        } else {
            $totalTax = (float) ($existing->total_tax ?? 0);
        }

        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $extra + $totalTax;

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        $this->saveTaxes($invoiceService, $request);
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        if (array_key_exists('tax_rate_id', $request)) {
            $this->saveTaxes($invoiceService, $request, true);
        }
    }

    private function saveTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (! is_array($taxRateIds)) {
            return;
        }

        if ($replace) {
            \App\Models\invoice\InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $dailyCost = (float) ($request['daily_cost'] ?? $invoiceService->daily_cost ?? 0);
        $subtotal = $dailyCost;

        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($subtotal * (float) $rate->percentage) / 100.0;
            \App\Models\invoice\InvoiceServicesTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'percentage' => (float) $rate->percentage,
                'tax_amount' => $amount,
            ]);
        }
    }
}
