<?php

namespace App\Http\Controllers\General;

use App\Http\Controllers\Controller;
use App\Http\Resources\CurrencyCollection;
use App\Http\Resources\CurrencyResource;
use App\Http\Resources\General\CurrencyHistoryResource;
use App\sys\ApiResponse;
use App\sys\Services\General\CurrencyService;
use Illuminate\Http\Request;

class CurrencyController extends Controller
{
    use ApiResponse;

    protected CurrencyService $currencyService;

    private $currencyTranslationService;

    public function __construct(CurrencyService $currencyService)
    {
        $this->currencyService = $currencyService;
        $this->currencyTranslationService = new \App\sys\Services\General\CurrencyTranslationService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currencies = $this->currencyService->getPaginatedCurrencies();

        return $this->apiResponse(200, 'success', null, new CurrencyCollection($currencies));
    }

    /**
     * Display the specified resource.
     */
    public function show(int $id)
    {
        $currency = $this->currencyService->getCurrencyById($id);

        if (! $currency) {
            $errors = $this->currencyService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CurrencyResource($currency));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $response = $this->currencyService->createCurrency($request->all());

        if (! $response) {
            $errors = $this->currencyService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CurrencyResource($response));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        $response = $this->currencyService->updateCurrency($request->all());

        if (! $response) {
            $errors = $this->currencyService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, new CurrencyResource($response));
    }

    public function getActive()
    {
        $data = $this->currencyService->getActive();
        $collect = new \App\Http\Resources\CurrencyCollection($data);

        return $this->apiResponse(200, 'Active Currencies retrieved successfully', null, $collect);
    }

    public function del(Request $request)
    {
        $ids = $request->input('ids', []);
        if (! is_array($ids) || empty($ids)) {
            return $this->apiResponse(400, 'No IDs provided for deletion', null, null);
        }
        $result = $this->currencyService->del($ids);
        if ($result) {
            return $this->apiResponse(200, 'Currencies deleted successfully', null, null);
        }
        $errors = $this->currencyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function saveTranslation(Request $request, $id)
    {
        if (! $this->currencyTranslationService->save($request->all(), $id)) {
            $errors = $this->currencyTranslationService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        return $this->apiResponse(200, 'success', null, null);
    }

    public function getTranslation($id)
    {
        if ($data = $this->currencyService->getByIdWithTranslation($id)) {
            $collection = new \App\Http\Resources\General\Currency\CurrencyLangResource($data);

            return $this->apiResponse(200, 'success', null, $collection);
        }
        $errors = $this->currencyService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function delLang($id)
    {
        if ($this->currencyTranslationService->del($id)) {
            return $this->apiResponse(200, 'success', null, null);
        }

        $errors = $this->currencyTranslationService->errors();

        return $this->apiResponse(400, 'fail', $errors, null);
    }

    public function getAllChanges($id)
    {
        $currency = $this->currencyService->getCurrencyById($id);
        if (! $currency) {
            $errors = $this->currencyService->errors();

            return $this->apiResponse(400, 'fail', $errors, null);
        }

        $changes = $this->currencyService->getAllChanges($currency);

        return $this->apiResponse(200, 'success', null, CurrencyHistoryResource::collection($changes));
    }

    public function getDefault()
    {
        $data = $this->currencyService->getDefault();
        return $this->apiResponse(200, 'Default Country retrieved successfully', null, new CurrencyResource($data));
    }
}
