<?php

namespace App\sys\Repository\General;

use App\Models\General\City;
use App\Models\General\Country;

class CityRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'country_name' => 'country_id',
        'country_id' => 'country_id',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return City::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('city_translation', function ($join) {
                    $join->on('cities.id', '=', 'city_translation.city_id')
                        ->where('city_translation.lang_id', app('lang_id'));
                })->orderBy('city_translation.name', $order)->select('cities.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return City::with('currentTranslation')->find($id);
    }

    public function create(array $data)
    {
        $city = new City;
        $city->country_id = $data['country_id'];
        $city->name = $data['name'];
        $city->shortcut = $data['shortcut'];
        $city->is_active = $data['active'];
        $city->save();

        return $city;
    }

    public function update(City $city, array $data)
    {
        $city->country_id = $data['country_id'] ?? $city->country_id;
        $city->name = $data['name'] ?? $city->name;
        $city->shortcut = $data['shortcut'] ?? $city->shortcut;
        $city->is_active = $data['active'] ?? $city->is_active;
        $city->save();

        return $city;
    }

    public function delete(City $city)
    {
        return $city->delete();
    }

    public function del(array $ids)
    {
        return \App\Models\General\City::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return City::with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return City::with('translations')->find($id);
    }

    public function getCitiesByCountry($countryId)
    {
        return City::with('currentTranslation')
            ->where('country_id', $countryId)
            ->where('is_active', 1)
            ->get();
    }

    public function getAllChanges($city)
    {
        return $city->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                $countryIds = array_filter([
                    $oldValues['country_id'] ?? null,
                    $newValues['country_id'] ?? null,
                ]);

                $countries = $countryIds ? Country::whereIn('id', $countryIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];

                if (isset($oldValues['country_id']) && isset($countries[$oldValues['country_id']])) {
                    $oldValues['country_name'] = $countries[$oldValues['country_id']];
                }
                if (isset($newValues['country_id']) && isset($countries[$newValues['country_id']])) {
                    $newValues['country_name'] = $countries[$newValues['country_id']];
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
