<?php

namespace App\sys\Repository\General;

use App\Models\General\Country;

class CountryRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'code' => 'code',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return Country::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('country_translation', function ($join) {
                    $join->on('countries.id', '=', 'country_translation.country_id')
                        ->where('country_translation.lang_id', app('lang_id'));
                })->orderBy('country_translation.name', $order)->select('countries.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Country::with('currentTranslation')->find($id);
    }

    public function create(array $data)
    {
        $country = new Country;
        $country->name = $data['name'];
        $country->code = $data['code'];
        $country->is_active = $data['active'];
        $country->save();

        return $country;
    }

    public function update(Country $country, array $data)
    {
        $country->name = $data['name'] ?? $country->name;
        $country->code = $data['code'] ?? $country->code;
        $country->is_active = $data['active'] ?? $country->is_active;
        $country->save();

        return $country;
    }

    public function delete(Country $country)
    {
        return $country->delete();
    }

    public function del(array $ids)
    {
        return \App\Models\General\Country::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Country::with('currentTranslation')->where('is_active', 1)->get();
    }



    public function getByIdWithTranslation($id)
    {
        return Country::with('translations')->find($id);
    }

    public function getAllChanges($country)
    {
        return $country->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
