<?php

namespace App\sys\Repository\General;

use App\Models\General\Currency;

class CurrencyRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'code' => 'code',
        'exchange_rate' => 'exchange_rate',
        'symbol' => 'symbol',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return Currency::when($name, function ($q, $name) {
            $q->whereHas('currentTranslation', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%");
            });
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('currency_translation', function ($join) {
                    $join->on('currencies.id', '=', 'currency_translation.currency_id')
                        ->where('currency_translation.lang_id', app('lang_id'));
                })->orderBy('currency_translation.name', $order)->select('currencies.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Currency::find($id);
    }

    public function create(array $data)
    {
        $currency = new Currency;
        $currency->name = $data['name'];
        $currency->symbol = $data['symbol'];
        $currency->exchange_rate = $data['exchange_rate'];
        $currency->status = $data['active'];
        $currency->code = $data['code'];
        $currency->save();

        return $currency;
    }

    public function changeDefault(int $id) {}

    public function update(Currency $currency, array $data)
    {
        $currency->name = $data['name'] ?? $currency->name;
        $currency->symbol = $data['symbol'] ?? $currency->symbol;
        $currency->exchange_rate = $data['exchange_rate'] ?? $currency->exchange_rate;
        $currency->status = $currency->is_default == 1 ? 1 : $data['active'] ?? $currency->status;
        $currency->code = $data['code'] ?? $currency->code;
        $currency->save();

        return $currency;
    }

    public function getActive()
    {
        return \App\Models\General\Currency::with('currentTranslation')->where('status', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return \App\Models\General\Currency::with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return \App\Models\General\Currency::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($currency)
    {
        return $currency->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
    public function getDefault()
    {
        return \App\Models\General\Currency::with('currentTranslation')->where('is_default', 1)->first();
    }
}
