<?php

namespace App\sys\Services\Accommodation;

use App\sys\Repository\Accommodation\ExtraRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class ExtraService extends Services
{
    protected ExtraRepository $extraRepository;

    public function __construct(ExtraRepository $extraRepository)
    {
        $this->extraRepository = $extraRepository;
    }

    public function getPaginatedExtras()
    {
        return $this->extraRepository->getPaginated();
    }

    public function getActive()
    {
        return $this->extraRepository->getActive();
    }

    public function getExtraById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_extras,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->extraRepository->findByIdOrFail($id);
    }

    public function createExtra($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $extra = $this->extraRepository->create($request);
        $this->syncTranslations($extra);

        return $extra;
    }

    public function updateExtra($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_extras,id'],
            'name' => ['sometimes', 'string', 'max:255'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $extra = $this->extraRepository->findByIdOrFail($request['id']);

        return $this->extraRepository->update($extra, $request);
    }

    public function deleteExtra(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_extras,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $extra = $this->extraRepository->findByIdOrFail($id);

        return $this->extraRepository->delete($extra);
    }

    public function del(array $ids)
    {
        return $this->extraRepository->del($ids);
    }

    public function getByidWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:ac_extras,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $extra = $this->extraRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($extra, true);
    }

    public function getAllChanges($extra)
    {
        return $this->extraRepository->getAllChanges($extra);
    }
}
