<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\RestaurantTranslationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class RestaurantTranslationService extends Services
{
    private $restaurantTranslation;

    public function __construct()
    {
        $this->restaurantTranslation = new RestaurantTranslationRepository;
    }

    public function save(array $data, $id)
    {
        $rules = ['restaurant_id' => 'required|integer|exists:ge_restaurant,id'];
        $combinations = [];
        $customErrors = [];

        foreach ($data as $index => $row) {
            $rowId = $row['id'] ?? null;
            $languageId = $row['language_id'] ?? null;
            $rules["$index.id"] = 'sometimes|nullable|exists:ge_restaurant_translation,id';
            $rules["$index.language_id"] = [
                Rule::requiredIf(empty($rowId)),
                'exists:languages,id',
                Rule::unique('ge_restaurant_translation', 'lang_id')
                    ->where(fn ($query) => $query->where('restaurant_id', $id))
                    ->ignore($rowId),
            ];
            $rules["$index.name"] = 'nullable|string|max:255';
            if ($languageId) {
                $key = $id.'-'.$languageId;
                if (in_array($key, $combinations)) {
                    $customErrors["$index.language_id"] = 'Same language distorted with the same Restaurant.';
                } else {
                    $combinations[] = $key;
                }
            }
        }
        $validator = Validator::make(array_merge($data, ['restaurant_id' => $id]), $rules);
        $validator->after(function ($validator) use ($customErrors) {
            foreach ($customErrors as $field => $message) {
                $validator->errors()->add($field, $message);
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->restaurantTranslation->save($data, $id);
    }

    public function get($id)
    {
        return $this->restaurantTranslation->get($id);
    }

    public function del($id)
    {
        $rules = [
            'id' => 'required|integer|exists:ge_restaurant_translation,id',
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->restaurantTranslation->del($id);
    }
}
