<?php

namespace App\sys\Services\General;

use App\Models\General\Currency;
use App\sys\Enums\Settings as SettingsEnum;
use App\sys\Helper;
use App\sys\Repository\General\SettingsRepository;
use App\sys\Services;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class SettingsService extends Services
{
    protected $settingsRepository;

    public function __construct(SettingsRepository $settingsRepository)
    {
        $this->settingsRepository = $settingsRepository;
    }

    /**
     * الحصول على جميع الإعدادات
     */
    public function getAllSettings()
    {
        return $this->settingsRepository->getAll();
    }

    /**
     * الحصول على إعداد محدد
     */
    public function getSetting($key)
    {
        return $this->settingsRepository->getByKey($key);
    }

    /**
     * تحديث إعداد محدد
     */
    public function updateSetting($data)
    {
        $validationRules = Helper::SettingsValidations();
        $validator = Validator::make($data, $validationRules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        foreach ($data as $key => $value) {
            if ($key == SettingsEnum::DEFAULT_CURRENCY->value) {
                return $this->updateDefaultCurrency($data['default_currency'] ?? null);
            }
            // تحديث إعداد عام
            $value = $data[$key] ?? null;
            if ($this->settingsRepository->exists($key)) {
                $this->settingsRepository->updateByKey($key, $value);
            } else {
                $this->settingsRepository->create($key, $value);
            }
        }

        // التحقق من القواعد الخاصة للـ default_currency
        return $this->settingsRepository->getAll();
    }

    /**
     * تحديث العملة الافتراضية مع التحقق من أنها لم تُحدد من قبل
     */
    public function updateDefaultCurrency($currencyId)
    {
        // التحقق من وجود العملة
        $currency = Currency::find($currencyId);
        if (! $currency) {
            throw new \Exception('العملة المحددة غير موجودة');
        }

        // التحقق من أن العملة الافتراضية لم تُحدد من قبل
        if ($this->settingsRepository->isDefaultCurrencySet()) {
            $currentDefaultCurrency = $this->settingsRepository->getDefaultCurrency();
            if ($currentDefaultCurrency != $currencyId) {
                throw new \Exception('لا يمكن تغيير العملة الافتراضية بعد تحديدها');
            }
        }

        // بدء المعاملة
        DB::beginTransaction();
        try {
            // تحديث أو إنشاء الإعداد
            if ($this->settingsRepository->exists(SettingsEnum::DEFAULT_CURRENCY->value)) {
                $this->settingsRepository->updateByKey(SettingsEnum::DEFAULT_CURRENCY->value, $currencyId);
            } else {
                $this->settingsRepository->create(SettingsEnum::DEFAULT_CURRENCY->value, $currencyId);
            }

            // إزالة is_default من جميع العملات الأخرى
            Currency::where('id', '!=', $currencyId)->update(['is_default' => 0]);

            // تعيين is_default = 1 للعملة المحددة
            $currency->update(['is_default' => 1]);

            DB::commit();

            return $this->settingsRepository->getByKey(SettingsEnum::DEFAULT_CURRENCY->value);

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}
