<?php

namespace App\sys\Services\Profile;

use App\sys\Repository\Profile\TravelerRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class TravelerService extends Services
{
    protected TravelerRepository $travelerRepository;

    public function __construct(TravelerRepository $travelerRepository)
    {
        $this->travelerRepository = $travelerRepository;
    }

    public function getPaginated($filters = [])
    {
        return $this->travelerRepository->getPaginated($filters);
    }

    public function getById($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->findByIdOrFail($id);
    }

    public function delete($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_profile_travelers,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $item = $this->travelerRepository->findByIdOrFail($id);

        return $this->travelerRepository->delete($item);
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_profile_travelers,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->del($ids);
    }

    public function createMultiple($request)
    {
        $request = $this->normalizeTravelersRequest($request);
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'travelers' => ['required', 'array', 'min:1'],
            'travelers.*.nationality_id' => ['required', 'integer', 'exists:nationalities,id'],
            'travelers.*.currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'travelers.*.type' => ['required', 'in:adult,child'],
            'travelers.*.unit_price' => ['required', 'numeric', 'min:0'],
            'travelers.*.count' => ['required', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->createMultiple($request);
    }

    public function updateMultiple($request)
    {
        $request = $this->normalizeTravelersRequest($request);
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'travelers' => ['required', 'array', 'min:1'],
            'travelers.*.nationality_id' => ['required', 'integer', 'exists:nationalities,id'],
            'travelers.*.currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'travelers.*.type' => ['required', 'in:adult,child'],
            'travelers.*.unit_price' => ['required', 'numeric', 'min:0'],
            'travelers.*.count' => ['required', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->updateMultiple($request);
    }

    private function normalizeTravelersRequest($request)
    {
        $travelers = [];
        if (isset($request['travelers']['adults']) && is_array($request['travelers']['adults'])) {
            foreach ($request['travelers']['adults'] as $traveler) {
                $traveler['type'] = 'adult';
                $travelers[] = $traveler;
            }
        }
        if (isset($request['travelers']['children']) && is_array($request['travelers']['children'])) {
            foreach ($request['travelers']['children'] as $traveler) {
                $traveler['type'] = 'child';
                $travelers[] = $traveler;
            }
        }
        $request['travelers'] = $travelers;

        return $request;
    }

    public function getAllChanges($profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->getAllChanges($profileId);
    }

    public function getTravelerNationalitiesByProfile($profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];

        $validator = Validator::make(['profile_id' => $profileId], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->travelerRepository->getTravelerNationalities($profileId);
    }
}
