<?php

namespace App\sys\Services\Suppliers;

use App\sys\Enums\SupplierAccountMappings;
use App\sys\Repository\Suppliers\SupplierRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class SupplierServices extends Services
{
    private $supplier;

    public function __construct()
    {
        $this->supplier = new SupplierRepository;
    }

    public function getByid($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:su_supplier,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->supplier->findByIdOrFail($id);
    }

    public function getLang($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:su_supplier,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $supplier = $this->supplier->getLang($id);

        return $this->syncTranslations($supplier, true);
    }

    public function active()
    {
        return $this->supplier->active();
    }

    public function del(array $ids)
    {
        return $this->supplier->del($ids);
    }

    public function get()
    {
        return $this->supplier->get();
    }

    public function add($data)
    {
        if (! $this->validAdd($data)) {
            return false;
        }
        $suppliers = $this->supplier->add($data);
        $this->syncTranslations($suppliers);

        return $suppliers;
    }

    public function updated($data)
    {
        if (! $this->validupadted($data)) {
            return false;
        }

        return $this->supplier->updated($data);
    }

    public function validAdd($data)
    {
        $rules = [
            'active' => ['required', 'integer', 'in:0,1'],
            'currency_id' => ['required', 'array'],
            'currency_id.*' => ['exists:currencies,id'],
            'services' => ['nullable', 'array'],
            'services.*' => ['exists:services,id'],
            'domains' => ['nullable', 'array'],
            'domains.*' => ['exists:domains,id'],
            'accommodations' => ['nullable', 'array'],
            'accommodations.*' => ['exists:ac_accommodations,id'],
            'supplier_name' => ['required', 'max:255', 'unique:su_supplier,supplier_name'],
            'supplier_vat' => ['required', 'max:255'],
            'supplier_address' => ['nullable', 'string'],
            'supplier_zip' => ['nullable', 'integer'],
            'supplier_email' => ['nullable', 'email'],
            'supplier_phone' => ['nullable', 'numeric'],
            'supplier_fax' => ['nullable', 'numeric'],
            'supplier_social_fb' => ['nullable'],
            'supplier_social_tw' => ['nullable'],
            'supplier_social_ld' => ['nullable'],
            'supplier_contact_person' => ['nullable', 'string'],
            'supplier_contact_phone' => ['nullable', 'numeric'],
            'supplier_contact_email' => ['nullable', 'email'],
            'payment_method' => ['required', 'in:wallet,credit,pay_on_booking'],
            'payment_method_accounting' => ['nullable', 'array'],
            'payment_method_accounting.*.input_name' => ['required_with:payment_method_accounting'],
            'payment_method_accounting.*.currency_id' => ['required_with:payment_method_accounting', 'exists:currencies,id'],
            'payment_method_accounting.*.balance' => ['required_with:payment_method_accounting', 'numeric'],
            //  'payment_method_accounting.*.tree_account_id' => ['required_with:payment_method_accounting', 'exists:tree_accounting,id'],
            'pay_on_booking' => ['nullable'],
            'accounting' => ['required', 'array'],
            'accounting.*.input_name' => ['required_with:accounting'],
            'accounting.*.currency_id' => ['required_with:accounting', 'exists:currencies,id'],
            'accounting.*.balance' => ['required_with:accounting', 'numeric'],
            'accounting.*.tree_account_id' => ['required_with:accounting', 'exists:tree_accounting,id'],
            'accounting.*.account_type' => ['required_with:accounting', 'in:credit,debit'],
        ];

        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            $method = $data['payment_method'] ?? null;
            if (in_array($method, ['wallet', 'credit'])) {
                if (empty($data['payment_method_accounting']) || ! is_array($data['payment_method_accounting'])) {
                    $validator->errors()->add('payment_method_accounting', 'This field is required when payment_method is wallet or credit.');
                }

                //  $paymentMethodUpper = strtoupper($method); // CREDIT or WALLET
                // $enumClass = "\\App\\sys\\Enums\\{$paymentMethodUpper}SupplierAccountMappings";
                // \App\sys\Helper::validateCurrencyInputCombinations($validator, $data, $enumClass, 'payment_method_accounting', $data['currency_id'], $paymentMethodUpper);
            }
            if ($method === 'pay_on_booking') {
                if (empty($data['pay_on_booking'])) {
                    $validator->errors()->add('pay_on_booking', 'This field is required when payment_method is pay_on_booking.');
                } elseif (! in_array($data['pay_on_booking'], ['employee_cash', 'company_cash'])) {
                    $validator->errors()->add('pay_on_booking', 'Invalid value for pay_on_booking.');
                }
            }
            if (! empty($data['accounting']) && is_array($data['accounting'])) {
                \App\sys\Helper::validateCurrencyInputCombinations($validator, $data, SupplierAccountMappings::class, 'accounting', $data['currency_id'], 'حسابات');
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    public function validupadted($data)
    {
        $rules = [
            'id' => ['required', 'exists:su_supplier,id'],
            'active' => ['sometimes', 'integer', 'in:0,1'],
            'currency_id' => ['sometimes', 'array'],
            'currency_id.*' => ['exists:currencies,id'],
            'services' => ['sometimes', 'array'],
            'services.*' => ['exists:services,id'],
            'category' => ['sometimes', 'array'],
            'category.*' => ['exists:categories,id'],
            'domains' => ['sometimes', 'nullable', 'array'],
            'domains.*' => ['exists:domains,id'],
            'supplier_name' => ['sometimes', 'max:255', 'unique:su_supplier,supplier_name,'.$data['id']],
            'supplier_vat' => ['sometimes', 'nullable', 'max:255'],
            'supplier_address' => ['sometimes', 'nullable', 'string'],
            'supplier_zip' => ['sometimes', 'nullable', 'integer'],
            'supplier_email' => ['sometimes', 'nullable', 'email'],
            'supplier_phone' => ['sometimes', 'nullable', 'numeric'],
            'supplier_fax' => ['sometimes', 'nullable', 'numeric'],
            'supplier_social_fb' => ['sometimes'],
            'supplier_social_tw' => ['sometimes'],
            'supplier_social_ld' => ['sometimes'],
            'supplier_contact_person' => ['sometimes', 'nullable', 'string'],
            'supplier_contact_phone' => ['sometimes', 'nullable', 'numeric'],
            'supplier_contact_email' => ['sometimes', 'nullable', 'email'],
            'payment_method' => ['sometimes', 'in:wallet,credit,pay_on_booking'],
            'payment_method_accounting' => ['sometimes', 'array'],
            'payment_method_accounting.*.input_name' => ['required_with:payment_method_accounting'],
            'payment_method_accounting.*.currency_id' => ['required_with:payment_method_accounting', 'exists:currencies,id'],
            'payment_method_accounting.*.balance' => ['required_with:payment_method_accounting', 'numeric'],
            // 'payment_method_accounting.*.tree_account_id' => ['required_with:payment_method_accounting', 'exists:tree_accounting,id'],
            'pay_on_booking' => ['sometimes'],
            'accounting' => ['sometimes', 'array'],
            'accounting.*.input_name' => ['required_with:accounting'],
            'accounting.*.currency_id' => ['required_with:accounting', 'exists:currencies,id'],
            'accounting.*.balance' => ['required_with:accounting', 'numeric'],
            'accounting.*.tree_account_id' => ['required_with:accounting', 'exists:tree_accounting,id'],
            'accounting.*.account_type' => ['required_with:accounting', 'in:credit,debit'],
        ];

        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            if (isset($data['payment_method'])) {
                $method = $data['payment_method'] ?? null;
                if (in_array($method, ['wallet', 'credit'])) {
                    if (empty($data['payment_method_accounting']) || ! is_array($data['payment_method_accounting'])) {
                        $validator->errors()->add('payment_method_accounting', 'This field is required when payment_method is wallet or credit.');
                    }
                    // $paymentMethodUpper = strtoupper($method); // CREDIT or WALLET
                    // $enumClass = "\\App\\sys\\Enums\\{$paymentMethodUpper}SupplierAccountMappings";
                    // \App\sys\Helper::validateCurrencyInputCombinations($validator, $data, $enumClass, 'payment_method_accounting', $data['currency_id'], $paymentMethodUpper);
                }
                if ($method === 'pay_on_booking') {
                    if (empty($data['pay_on_booking'])) {
                        $validator->errors()->add('pay_on_booking', 'This field is required when payment_method is pay_on_booking.');
                    } elseif (! in_array($data['pay_on_booking'], ['employee_cash', 'company_cash'])) {
                        $validator->errors()->add('pay_on_booking', 'Invalid value for pay_on_booking.');
                    }
                }
            }

            if (! empty($data['accounting']) && is_array($data['accounting'])) {
                \App\sys\Helper::validateCurrencyInputCombinations($validator, $data, SupplierAccountMappings::class, 'accounting', $data['currency_id'], 'حسابات');
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    public function getCurrencyWithtree($data)
    {
        return \App\sys\Helper::getCurrencyWithtree(SupplierAccountMappings::class, $data);
    }

    public function getCurrencyWithtreeByPaymentMethods($data)
    {
        $rules = [
            'currency_id' => ['required', 'array'],
            'currency_id.*' => ['exists:currencies,id'],
            'payment_methods' => ['required', 'in:credit,wallet'],
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $paymentMethodUpper = strtoupper($data['payment_methods']);
        $enumsName = '\\App\\sys\\Enums\\'.$paymentMethodUpper.'SupplierAccountMappings';

        return \App\sys\Helper::getCurrencyWithtree($enumsName, $data['currency_id']);
    }

    public function supplierByServiceAndCurrency($serviceId, $currencyId)
    {
        $rules = [
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
        ];
        $validator = Validator::make(['service_id' => $serviceId, 'currency_id' => $currencyId], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
        }

        return $this->supplier->supplierByServiceAndCurrency($serviceId, $currencyId);
    }

    public function suppliersByAccommodationId($accommodationId)
    {
        $rules = [
            'accommodation_id' => ['required', 'integer', 'exists:ac_accommodations,id'],
        ];
        $validator = Validator::make(['accommodation_id' => $accommodationId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $suppliers = $this->supplier->suppliersByAccommodationId((int) $accommodationId);

        return $suppliers->map(function ($supplier) {
            return [
                'id' => $supplier->id,
                'supplier_name' => $supplier->currentTranslation->supplier_name ?? $supplier->supplier_name,
                'currency' => $supplier->accounting
                    ->pluck('currency')
                    ->filter()
                    ->map(function ($currency) {
                        return $currency->currentTranslation->name ?? $currency->name;
                    })
                    ->unique()
                    ->values()
                    ->toArray(),
            ];
        });
    }
}
