<?php

namespace App\sys\Repository\Accommodation;

use App\Models\Accommodation\Accommodations;

class AccommodationsRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'description' => 'description',
        'type' => 'type',
        'rating_id' => 'rating_id',
        'country_id' => 'country_id',
        'city_id' => 'city_id',
        'active' => 'is_active',
        'address' => 'address',
        'phone' => 'phone',
        'email' => 'email',
        'website' => 'website',
        'taxable' => 'taxable',
        'is_main_provider' => 'is_main_provider',
        'created_at' => 'created',
        'updated_at' => 'modified',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);
        $type = request('type', null);

        return Accommodations::when($name, function ($q, $name) {
            $q->whereHas('translations', function ($q) use ($name) {
                $q->where('name', 'LIKE', "%$name%")->where('lang_id', app('lang_id'));
            });
        })->when($type, function ($q, $type) {
            $q->where('type', $type);
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            $query->orderBy($this->columns[$column], $order);
        })->with(['currentTranslation', 'country.currentTranslation', 'city.currentTranslation', 'rating.currentTranslation'])->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Accommodations::with(['currentTranslation', 'country.currentTranslation', 'city.currentTranslation', 'rating.currentTranslation', 'suppliers.currentTranslation'])->find($id);
    }

    public function create(array $data)
    {
        $item = new Accommodations;
        $item->name = $data['name'];
        $item->description = $data['description'];
        $item->type = $data['type'];
        $item->rating_id = $data['rating_id'];
        $item->country_id = $data['country_id'];
        $item->city_id = $data['city_id'];
        $item->is_active = $data['active'];
        $item->address = $data['address'];
        $item->phone = $data['phone'];
        $item->email = $data['email'];
        $item->website = $data['website'];
        $item->taxable = $data['taxable'];
        $item->is_main_provider = $data['is_main_provider'];
        $item->save();

        // Handle supplier relationships
        if (isset($data['supplier_ids']) && is_array($data['supplier_ids'])) {
            $item->suppliers()->attach($data['supplier_ids']);
        }

        // Load the item with suppliers for response
        return $item->load('suppliers.currentTranslation');
    }

    public function update(Accommodations $item, array $data)
    {
        $item->name = $data['name'] ?? $item->name;
        $item->description = $data['description'] ?? $item->description;
        $item->type = $data['type'] ?? $item->type;
        $item->rating_id = $data['rating_id'] ?? $item->rating_id;
        $item->country_id = $data['country_id'] ?? $item->country_id;
        $item->city_id = $data['city_id'] ?? $item->city_id;
        $item->is_active = $data['active'] ?? $item->is_active;
        $item->address = $data['address'] ?? $item->address;
        $item->phone = $data['phone'] ?? $item->phone;
        $item->email = $data['email'] ?? $item->email;
        $item->website = $data['website'] ?? $item->website;
        $item->taxable = $data['taxable'] ?? $item->taxable;
        $item->is_main_provider = $data['is_main_provider'] ?? $item->is_main_provider;
        $item->save();

        // Handle supplier relationships
        if (isset($data['supplier_ids'])) {
            if (is_array($data['supplier_ids']) && ! empty($data['supplier_ids'])) {
                $item->suppliers()->sync($data['supplier_ids']);
            } else {
                $item->suppliers()->detach();
            }
        }

        // Load the item with suppliers for response
        return $item->load('suppliers.currentTranslation');
    }

    public function delete(Accommodations $item)
    {
        return $item->delete();
    }

    public function del(array $ids)
    {
        return Accommodations::whereIn('id', $ids)->delete();
    }

    public function getActive()
    {
        return Accommodations::with(['currentTranslation', 'country.currentTranslation', 'city.currentTranslation', 'rating.currentTranslation'])->where('is_active', 1)->get();
    }

    public function getByType(string $type, $city_id)
    {
        return Accommodations::with(['currentTranslation', 'country.currentTranslation', 'city.currentTranslation', 'rating.currentTranslation'])
            ->where('type', $type)
            ->where('is_active', 1)
            ->where('city_id', $city_id)
            ->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Accommodations::with('translations')->find($id);
    }

    public function getAllChanges($accommodation)
    {
        return $accommodation->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                // Get country IDs
                $countryIds = array_filter([
                    $oldValues['country_id'] ?? null,
                    $newValues['country_id'] ?? null,
                ]);

                // Get city IDs
                $cityIds = array_filter([
                    $oldValues['city_id'] ?? null,
                    $newValues['city_id'] ?? null,
                ]);

                // Get rating IDs
                $ratingIds = array_filter([
                    $oldValues['rating_id'] ?? null,
                    $newValues['rating_id'] ?? null,
                ]);

                // Get countries
                $countries = $countryIds ? \App\Models\General\Country::whereIn('id', $countryIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];

                // Get cities
                $cities = $cityIds ? \App\Models\General\City::whereIn('id', $cityIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];

                // Get ratings
                $ratings = $ratingIds ? \App\Models\Accommodation\Rating::whereIn('id', $ratingIds)
                    ->pluck('name', 'id')
                    ->toArray() : [];

                // Add country names
                if (isset($oldValues['country_id']) && isset($countries[$oldValues['country_id']])) {
                    $oldValues['country_name'] = $countries[$oldValues['country_id']];
                }
                if (isset($newValues['country_id']) && isset($countries[$newValues['country_id']])) {
                    $newValues['country_name'] = $countries[$newValues['country_id']];
                }

                // Add city names
                if (isset($oldValues['city_id']) && isset($cities[$oldValues['city_id']])) {
                    $oldValues['city_name'] = $cities[$oldValues['city_id']];
                }
                if (isset($newValues['city_id']) && isset($cities[$newValues['city_id']])) {
                    $newValues['city_name'] = $cities[$newValues['city_id']];
                }

                // Add rating names
                if (isset($oldValues['rating_id']) && isset($ratings[$oldValues['rating_id']])) {
                    $oldValues['rating_name'] = $ratings[$oldValues['rating_id']];
                }
                if (isset($newValues['rating_id']) && isset($ratings[$newValues['rating_id']])) {
                    $newValues['rating_name'] = $ratings[$newValues['rating_id']];
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
