<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\Paytype;
use App\Models\Accounting\PaytypeAccounting;
use App\Models\Accounting\PaytypeTranslation;

class PaytypeRepository
{
    private $columns = [
        'id' => 'id',
        'title' => 'title',
        'percentage' => 'percentage',
        'treasury_bank_later' => 'treasury_bank_later',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $title = request('name', null);
        $limit = request('limit', 15);

        $query = Paytype::query();

        // Subquery يرجّع الترجمة الحالية
        $translationSub = PaytypeTranslation::select('title')
            ->whereColumn('pay_type_id', 'pay_type.id')
            ->where('lang_id', app('lang_id'))
            ->limit(1);

        $query->addSelect([
            'title' => $translationSub,
        ]);

        // فلترة بالعنوان (على الترجمة)
        $query->when($title, function ($q, $title) {
            $q->where('title', 'LIKE', "%$title%");
        });

        // ترتيب
        $query->when($column && array_key_exists($column, $this->columns), function ($q) use ($column, $order) {
            if ($column === 'title') {
                $q->orderBy('title', $order);
            } else {
                $q->orderBy($this->columns[$column], $order);
            }
        });

        // نجيب العلاقة currentTranslation كمان
        return $query->with(['currentTranslation' => function ($q) {
            $q->select('id', 'pay_type_id', 'title', 'lang_id');
        }])->paginate($limit);
    }

    public function getByCurrnecy($currnecy)
    {
        return Paytype::where(function ($q) use ($currnecy) {
            $q->where('is_protected', 0)
                ->orWhereHas('accounting', function ($q2) use ($currnecy) {
                    $q2->where('currency_id', $currnecy);
                });
        })->with(['accounting' => function ($q) use ($currnecy) {
            $q->where('currency_id', $currnecy);
        }])->get();
    }

    public function findByIdOrFail(int $id)
    {
        return Paytype::with('accounting')->find($id);
    }

    public function getpayTypeAccountingByCurrency($pay_type_id, $currency)
    {
        return PaytypeAccounting::where([
            ['pay_type_id', $pay_type_id],
            ['currency_id', $currency],
        ])->first();
    }

    public function getByIdWithTranslation($id)
    {
        return Paytype::with('translations')->where('id', $id)->first();
    }

    public function create(array $data)
    {
        $paytype = new Paytype;
        $paytype->title = $data['title'];
        $paytype->is_protected = $data['is_protected'] ?? false;
        $paytype->save();

        $this->saveAccounting($data, $paytype->id);

        return $paytype->load('accounting');
    }

    public function update($data)
    {
        $paytype = Paytype::find($data['id']);

        if (! $paytype) {
            return false;
        }

        $paytype->title = $data['title'] ?? $paytype->title;
        $paytype->is_protected = $data['is_protected'] ?? $paytype->is_protected;
        $paytype->save();

        $this->updatedAccounting($data, $paytype->id);

        return $paytype->load('accounting');
    }

    private function updatedAccounting($data, $id)
    {
        if (isset($data['accounting'])) {
            foreach ($data['accounting'] as $value) {
                if (PaytypeAccounting::where(['pay_type_id' => $id, 'currency_id' => $value['currency_id']])->exists()) {
                    $this->updatethisAccount($value, $id);
                } else {
                    $this->addAccounting($value, $id);
                }
            }
        }
    }

    public function updatethisAccount($data, $paytype_id)
    {
        $account = PaytypeAccounting::where(['pay_type_id' => $paytype_id, 'currency_id' => $data['currency_id']])->first();

        if (! $account) {
            return false;
        }
        $account->percentage = $data['percentage'];
        $account->tree_accounting_transfer_id = $data['tree_accounting_transfer_id'];
        $account->tree_account_commission_id = $data['tree_account_commission_id'];
        $account->min_amount = $data['min_amount'] ?? 0;
        $account->max_amount = $data['max_amount'] ?? 0;
        $account->save();
    }

    public function saveAccounting($data, $paytype_id)
    {
        foreach ($data['accounting'] as $value) {
            $this->addAccounting($value, $paytype_id);
        }
    }

    public function addAccounting($data, $paytype_id)
    {
        $account = new PaytypeAccounting;
        $account->pay_type_id = $paytype_id;
        $account->currency_id = $data['currency_id'];
        $account->percentage = $data['percentage'];
        $account->tree_accounting_transfer_id = $data['tree_accounting_transfer_id'];
        $account->tree_account_commission_id = $data['tree_account_commission_id'];
        $account->min_amount = $data['min_amount'] ?? 0;
        $account->max_amount = $data['max_amount'] ?? 0;
        $account->save();
    }

    public function del(array $ids)
    {
        // Check for protected records
        $protectedRecords = Paytype::whereIn('id', $ids)->where('is_protected', true)->get();

        if ($protectedRecords->count() > 0) {
            $protectedTitles = $protectedRecords->pluck('title')->toArray();

            return [
                'success' => false,
                'message' => 'لا يمكن حذف السجلات المحمية: '.implode(', ', $protectedTitles),
                'protected_records' => $protectedTitles,
            ];
        }

        $deleted = Paytype::whereIn('id', $ids)->delete();

        return [
            'success' => true,
            'deleted_count' => $deleted,
        ];
    }

    public function getAllChanges($paytype)
    {
        // Get paytype changes
        $paytypeChanges = $paytype->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                    'audit_type' => 'paytype',
                ];
            });

        // Get accounting mapping changes
        $accountingChanges = PaytypeAccounting::where('pay_type_id', $paytype->id)
            ->with(['audits' => function ($query) {
                $query->with(['user' => function ($userQuery) {
                    $userQuery->select('id', 'name');
                }]);
            }])
            ->get()
            ->flatMap(function ($accountingMapping) {
                return $accountingMapping->audits->map(function ($audit) {
                    $oldValues = $audit->old_values ?? [];
                    $newValues = $audit->new_values ?? [];

                    return [
                        'audit_id' => $audit->id,
                        'user_id' => $audit->user_id ?? null,
                        'user' => $audit->user ? $audit->user->toArray() : null,
                        'old_values' => $oldValues,
                        'new_values' => $newValues,
                        'changed_at' => $audit->created_at,
                        'event' => $audit->event,
                        'ip_address' => $audit->ip_address,
                        'user_agent' => $audit->user_agent,
                        'audit_type' => 'accounting_mapping',
                        'accounting_mapping_id' => $audit->auditable_id,
                    ];
                });
            });

        // Combine and sort all changes by timestamp
        $allChanges = $paytypeChanges->concat($accountingChanges)
            ->sortByDesc('changed_at')
            ->values();

        return $allChanges;
    }
}
