<?php

namespace App\sys\Repository\Accounting;

use App\Models\Accounting\CostCenter;
use App\Models\Accounting\Transfer;
use App\Models\Accounting\TreeAccounting;
use App\Models\General\Currency;

class ReportRepository
{
    public function getReports()
    {
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $transfers = Transfer::with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->whereDate('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            })->get();

        return $this->formatTransferResponse($transfers, $currency, $defaultCurrency);
    }

    public function getBalanceAndIncomeReport($reportType)
    {
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        if ($reportType === 'balance') {
            $types = ['assets', 'liabilities'];
        } elseif ($reportType === 'income') {
            $types = ['expenditure', 'revenue'];
        }

        $treeAccounts = TreeAccounting::whereIn('id', function ($query) use ($types) {
            $query->select('tree_account_id')
                ->from('tree_account_settings')
                ->whereIn('type', $types);
        })
            ->with('getAllChildren')
            ->get();

        $treeAccountingIds = [];

        foreach ($treeAccounts as $account) {
            $treeAccountingIds[] = $account->id;
            $treeAccountingIds = array_merge($treeAccountingIds, $account->getAllChildrenIdsFromRelationship());
        }
        $treeAccountingIds = array_values(array_unique($treeAccountingIds));

        $transfers = Transfer::whereIn('tree_accounting_id', $treeAccountingIds)->with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->where('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->where('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            })
            ->get();

        return $this->formatTransferResponse($transfers, $currency, $defaultCurrency);
    }

    private function formatTransferResponse($transfers, $currency, $defaultCurrency)
    {
        if ($currency == null) {
            return [
                'data' => $transfers->map(function ($transfer) {
                    return [
                        'id' => $transfer->id,
                        'constraint_number_doc' => $transfer->constraint->number_doc ?? null,
                        'transfer_name' => $transfer->name,
                        'creditor' => $transfer->debit,
                        'debit' => $transfer->creditor,
                        'date' => $transfer->date,
                        'type' => $transfer->constraint->capture_exchange ?? null,
                        'constraint_name' => $transfer->constraint->name ?? null,
                        'constraint_description' => $transfer->constraint->description ?? null,
                        'tree_accounting_title' => $transfer->treeAccounting->currentTranslation->title ?? ($transfer->treeAccounting->title ?? null),
                        'tree_accounting_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                        'currency_name' => $transfer->currency->name ?? null,
                        'currency_code' => $transfer->currency->code ?? null,
                        'currency_symbol' => $transfer->currency->symbol ?? null,
                        'currency_transfer_rate' => $transfer->currency_transfer_rate ?? null,
                    ];
                }),
                'summery' => [
                    'total_debit' => $transfers->sum('debit'),
                    'total_creditor' => $transfers->sum('creditor'),
                ],
            ];
        } else {
            $totalCreditAfterExchange = $transfers->sum(function ($transfer) {
                return $transfer->currency_creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitAfterExchange = $transfers->sum(function ($transfer) {
                return $transfer->currency_debit * ($transfer->currency_transfer_rate ?? 1);
            });

            return [
                'data' => $transfers->map(function ($transfer) use ($defaultCurrency) {
                    return [
                        'id' => $transfer->id,
                        'constraint_number_doc' => $transfer->constraint->number_doc ?? null,
                        'transfer_name' => $transfer->name,
                        'creditor' => $transfer->currency_creditor,
                        'debit' => $transfer->currency_debit,
                        'date' => $transfer->date,
                        'type' => $transfer->constraint->capture_exchange ?? null,
                        'constraint_name' => $transfer->constraint->name ?? null,
                        'constraint_description' => $transfer->constraint->description ?? null,
                        'tree_accounting_title' => $transfer->treeAccounting->currentTranslation->title ?? ($transfer->treeAccounting->title ?? null),
                        'tree_accounting_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                        'currency_name' => $transfer->currency->name ?? null,
                        'currency_code' => $transfer->currency->code ?? null,
                        'currency_symbol' => $transfer->currency->symbol ?? null,
                        'currency_transfer_rate' => $transfer->currency_transfer_rate ?? null,
                        'default_currency_name' => $defaultCurrency->name ?? null,
                        'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                        'default_currency_code' => $defaultCurrency->code ?? null,
                        'default_currency_exchange_rate' => $defaultCurrency->exchange_rate ?? null,
                    ];
                }),
                'summery' => [
                    'total_debit' => $totalDebitAfterExchange,
                    'total_creditor' => $totalCreditAfterExchange,
                ],
            ];
        }
    }

    public function getHierarchicalTreeAccountReport()
    {
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from', null);
        $dateTo = request('date_to', null);
        $type = request('type', 'balance');

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $types = ($type == 'balance') ? ['assets', 'liabilities'] : ['revenue', 'expenditure'];

        $mainTreeAccounts = TreeAccounting::where('tree_accounting_id', 0)
            ->where('active', 1)
            ->whereIn('id', function ($query) use ($types) {
                $query->select('tree_account_id')
                    ->from('tree_account_settings')
                    ->whereIn('type', $types);
            })
            ->with(['getAllChildren'])
            ->get();

        $result = [];
        $grandTotalBalance = 0;

        foreach ($mainTreeAccounts as $mainAccount) {
            $accountData = $this->buildHierarchicalAccountData(
                $mainAccount,
                $currency,
                $companyIds,
                $dateFrom,
                $dateTo,
                $defaultCurrency
            );

            if ($accountData) {
                // Get the account type from tree_account_settings
                $accountType = $mainAccount->settings()->whereIn('type', $types)->value('type');

                // Adjust grand total based on type
                if ($type == 'balance') {
                    if ($accountType == 'assets') {
                        $grandTotalBalance += $accountData['balance'];
                    } elseif ($accountType == 'liabilities') {
                        $grandTotalBalance -= $accountData['balance'];
                    }
                } else {
                    if ($accountType == 'revenue') {
                        $grandTotalBalance += $accountData['balance'];
                    } elseif ($accountType == 'expenditure') {
                        $grandTotalBalance -= $accountData['balance'];
                    }
                }

                $result[] = $accountData;
            }
        }

        return [
            'data' => $result,
            'totals' => [
                'grand_total_balance' => $grandTotalBalance,
            ],
            'currency_info' => $currency ? null : [
                'default_currency_name' => $defaultCurrency->name ?? null,
                'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                'default_currency_code' => $defaultCurrency->code ?? null,
            ],
        ];
    }

    private function buildHierarchicalAccountData($account, $currency, $companyIds, $dateFrom, $dateTo, $defaultCurrency)
    {
        // Get all child account IDs for this account
        $allChildIds = [$account->id];
        if ($account->getAllChildren) {
            $allChildIds = array_merge($allChildIds, $account->getAllChildrenIdsFromRelationship());
        }

        // Get transfers for this account and all its children
        $transfersQuery = Transfer::whereIn('tree_accounting_id', $allChildIds)
            ->with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($dateFrom, function ($q, $dateFrom) {
                $q->whereDate('date', '>=', $dateFrom);
            })
            ->when($dateTo, function ($q, $dateTo) {
                $q->whereDate('date', '<=', $dateTo);
            })
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            });

        $transfers = $transfersQuery->get();

        // Calculate totals
        $totalCreditor = 0;
        $totalDebit = 0;

        if ($currency) {
            // If currency is specified, use direct amounts
            $totalCreditor = $transfers->sum('creditor');
            $totalDebit = $transfers->sum('debit');
        } else {
            // If no currency specified, convert to default currency
            $totalCreditor = $transfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebit = $transfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        // Build children data
        $children = [];
        if ($account->getAllChildren) {
            foreach ($account->getAllChildren as $child) {
                $childData = $this->buildHierarchicalAccountData(
                    $child,
                    $currency,
                    $companyIds,
                    $dateFrom,
                    $dateTo,
                    $defaultCurrency
                );

                if ($childData) {
                    $children[] = $childData;
                }
            }
        }

        // Calculate total from children
        $childrenTotalCreditor = array_sum(array_column($children, 'total_creditor'));
        $childrenTotalDebit = array_sum(array_column($children, 'total_debit'));

        // If this account has no direct transfers but has children, use children totals
        if ($transfers->isEmpty() && ! empty($children)) {
            $totalCreditor = $childrenTotalCreditor;
            $totalDebit = $childrenTotalDebit;
        }

        return [
            'id' => $account->id,
            'title' => $account->title,
            'the_level' => $account->the_level,
            'serial_number' => $account->serial_number,
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'balance' => $totalDebit - $totalCreditor, // Debit - Credit
            'children' => $children,
            'has_children' => ! empty($children),
            'direct_transfers_count' => $transfers->count(),
        ];
    }

    public function getHierarchicalTreeAccountReportTrailBalance()
    {
        $currency = request('currency_id', null);
        $companyIds = request('company_ids', []);
        $dateFrom = request('date_from');
        $dateTo = request('date_to');

        if (is_string($companyIds)) {
            $companyIds = json_decode($companyIds, true) ?? [];
        }
        $companyIds = is_array($companyIds) ? $companyIds : [];

        $defaultCurrency = Currency::where('is_default', 1)->first();

        $types = ['assets', 'liabilities', 'revenue', 'expenditure'];

        $mainTreeAccounts = TreeAccounting::where('tree_accounting_id', 0)
            ->where('active', 1)

            ->with(['getAllChildren'])
            ->get();

        $result = [];
        $grandTotals = [
            'total_creditor_initial' => 0,
            'total_debit_initial' => 0,
            'total_creditor_within' => 0,
            'total_debit_within' => 0,
            'total_creditor' => 0,
            'total_debit' => 0,
            'balance_initial' => 0,
            'balance_within' => 0,
            'balance_total' => 0,
        ];

        foreach ($mainTreeAccounts as $mainAccount) {
            $accountData = $this->buildHierarchicalAccountDataWithInitial(
                $mainAccount,
                $currency,
                $companyIds,
                $dateFrom,
                $dateTo,
                $defaultCurrency
            );

            if ($accountData) {
                // Get the account type from tree_account_settings
                $accountType = $mainAccount->settings()->whereIn('type', $types)->value('type');
                $accountData['account_type'] = $accountType;

                // Add to grand totals
                $grandTotals['total_creditor_initial'] += $accountData['total_creditor_initial'];
                $grandTotals['total_debit_initial'] += $accountData['total_debit_initial'];
                $grandTotals['total_creditor_within'] += $accountData['total_creditor_within'];
                $grandTotals['total_debit_within'] += $accountData['total_debit_within'];
                $grandTotals['total_creditor'] += $accountData['total_creditor'];
                $grandTotals['total_debit'] += $accountData['total_debit'];

                // Calculate balances based on account type
                if (in_array($accountType, ['assets', 'expenditure'])) {
                    $grandTotals['balance_initial'] += $accountData['balance_initial'];
                    $grandTotals['balance_within'] += $accountData['balance_within'];
                    $grandTotals['balance_total'] += $accountData['balance_total'];
                } else {
                    $grandTotals['balance_initial'] -= $accountData['balance_initial'];
                    $grandTotals['balance_within'] -= $accountData['balance_within'];
                    $grandTotals['balance_total'] -= $accountData['balance_total'];
                }

                $result[] = $accountData;
            }
        }

        return [
            'data' => $result,
            'totals' => $grandTotals,
            'currency_info' => $currency ? null : [
                'default_currency_name' => $defaultCurrency->name ?? null,
                'default_currency_symbol' => $defaultCurrency->symbol ?? null,
                'default_currency_code' => $defaultCurrency->code ?? null,
            ],
            'period_info' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
        ];
    }

    private function buildHierarchicalAccountDataWithInitial($account, $currency, $companyIds, $dateFrom, $dateTo, $defaultCurrency)
    {
        $allChildIds = [$account->id];
        if ($account->getAllChildren) {
            $allChildIds = array_merge($allChildIds, $account->getAllChildrenIdsFromRelationship());
        }

        $baseTransfersQuery = Transfer::whereIn('tree_accounting_id', $allChildIds)
            ->with('constraint', 'treeAccounting.currentTranslation', 'currency')
            ->when($currency, function ($q, $currency) {
                $q->where('currency_id', $currency);
            })
            ->when(! empty($companyIds), function ($q) use ($companyIds) {
                $q->whereHas('constraint', function ($query) use ($companyIds) {
                    $query->whereIn('company_id', $companyIds);
                });
            });

        $initialTransfersQuery = clone $baseTransfersQuery;
        $initialTransfers = $initialTransfersQuery->whereDate('date', '<', $dateFrom)->get();

        $withinTransfersQuery = clone $baseTransfersQuery;
        $withinTransfers = $withinTransfersQuery
            ->whereDate('date', '>=', $dateFrom)
            ->whereDate('date', '<=', $dateTo)
            ->get();

        $totalCreditorInitial = 0;
        $totalDebitInitial = 0;

        if ($currency) {
            $totalCreditorInitial = $initialTransfers->sum('creditor');
            $totalDebitInitial = $initialTransfers->sum('debit');
        } else {
            $totalCreditorInitial = $initialTransfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitInitial = $initialTransfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        $totalCreditorWithin = 0;
        $totalDebitWithin = 0;

        if ($currency) {
            $totalCreditorWithin = $withinTransfers->sum('creditor');
            $totalDebitWithin = $withinTransfers->sum('debit');
        } else {
            $totalCreditorWithin = $withinTransfers->sum(function ($transfer) {
                return $transfer->creditor * ($transfer->currency_transfer_rate ?? 1);
            });
            $totalDebitWithin = $withinTransfers->sum(function ($transfer) {
                return $transfer->debit * ($transfer->currency_transfer_rate ?? 1);
            });
        }

        $children = [];
        if ($account->getAllChildren) {
            foreach ($account->getAllChildren as $child) {
                $childData = $this->buildHierarchicalAccountDataWithInitial(
                    $child,
                    $currency,
                    $companyIds,
                    $dateFrom,
                    $dateTo,
                    $defaultCurrency
                );

                if ($childData) {
                    $children[] = $childData;
                }
            }
        }

        $childrenCreditorInitial = array_sum(array_column($children, 'total_creditor_initial'));
        $childrenDebitInitial = array_sum(array_column($children, 'total_debit_initial'));
        $childrenCreditorWithin = array_sum(array_column($children, 'total_creditor_within'));
        $childrenDebitWithin = array_sum(array_column($children, 'total_debit_within'));

        if ($initialTransfers->isEmpty() && $withinTransfers->isEmpty() && ! empty($children)) {
            $totalCreditorInitial = $childrenCreditorInitial;
            $totalDebitInitial = $childrenDebitInitial;
            $totalCreditorWithin = $childrenCreditorWithin;
            $totalDebitWithin = $childrenDebitWithin;
        }

        $totalCreditor = $totalCreditorInitial + $totalCreditorWithin;
        $totalDebit = $totalDebitInitial + $totalDebitWithin;

        $balanceInitial = $totalDebitInitial - $totalCreditorInitial;
        $balanceWithin = $totalDebitWithin - $totalCreditorWithin;
        $balanceTotal = $totalDebit - $totalCreditor;

        return [
            'id' => $account->id,
            'title' => $account->title,
            'the_level' => $account->the_level,
            'serial_number' => $account->serial_number,

            // Initial amounts (before start date)
            'total_creditor_initial' => $totalCreditorInitial,
            'total_debit_initial' => $totalDebitInitial,
            'balance_initial' => $balanceInitial,

            // Within period amounts
            'total_creditor_within' => $totalCreditorWithin,
            'total_debit_within' => $totalDebitWithin,
            'balance_within' => $balanceWithin,

            // Total amounts (initial + within)
            'total_creditor' => $totalCreditor,
            'total_debit' => $totalDebit,
            'balance_total' => $balanceTotal,

            'children' => $children,
            'has_children' => ! empty($children),
            'initial_transfers_count' => $initialTransfers->count(),
            'within_transfers_count' => $withinTransfers->count(),
            'total_transfers_count' => $initialTransfers->count() + $withinTransfers->count(),
        ];
    }

    public function treeAccountTransfer(array $request)
    {

        $trees = TreeAccounting::whereIn('id', $request['treeAccount_ids'] ?? [])
            ->with([
                'currentTranslation',
                'transfers' => function ($query) use ($request) {
                    $query->when($request['date_from'] ?? null, function ($query) use ($request) {
                        return $query->where('date', '>=', $request['date_from']);
                    })
                        ->when($request['date_to'] ?? null, function ($query) use ($request) {
                            return $query->where('date', '<=', $request['date_to']);
                        })
                        ->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                            $query->whereHas('constraint', function ($subQuery) use ($request) {
                                $subQuery->whereIn('company_id', $request['company_ids']);
                            });
                        });
                },
                'transfers.constraint',
                'transfers.costCenter',
                'transfers.currency.currentTranslation',
                'transfers.treeAccounting',
            ])->get();

        return $trees->map(function ($tree) use ($request) {
            $account['currency_id'] = $request['show_by_default']?null:$tree->currency_id;
            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;
            $totalBalance = 0;

            foreach ($tree->transfers as $transfer) {

                $debit = $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                $credit = $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                $balance = $debit - $credit;
                $totalBalance += $balance;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'cost_center_name' => $transfer->costCenter->name ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'tree_account_title' => $transfer->treeAccounting->title ?? null,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'debit' => $debit,
                    'credit' => $credit,
                    'balance' => $totalBalance,
                    'rate' => $transfer->currency_transfer_rate,
                    'running_balance' => $totalBalance,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'tree_account_id' => $tree->id,
                'tree_account_title' => $tree->title,
                'tree_account_serial_number' => $tree->serial_number,
                'transactions' => $transactions,
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'total_balance' => $totalDebit - $totalCredit,
                'show_by_default' => $account['currency_id'] == null ? true : false,

            ];
        });
    }

    public function treeAccountTransferWithInitialBalance(array $request)
    {

        $trees = TreeAccounting::whereIn('id', $request['treeAccount_ids'] ?? [])
            ->with([
                'currentTranslation',
                'transfers' => function ($query) use ($request) {
                    $query->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                        $query->whereHas('constraint', function ($subQuery) use ($request) {
                            $subQuery->whereIn('company_id', $request['company_ids']);
                        });
                    });
                },
                'transfers.constraint',
                'transfers.costCenter',
                'transfers.currency.currentTranslation',
            ])
            ->get();

        return $trees->map(function ($tree) use ($request) {

            $account['currency_id']=$request['show_by_default']?null:$tree->currency_id;

            // Calculate initial balance from transfers before start date
            $initialDebit = 0;
            $initialCredit = 0;

            if (! empty($request['date_from'])) {
                foreach ($tree->transfers as $transfer) {
                    if ($transfer->date < $request['date_from']) {
                        $initialDebit += $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                        $initialCredit += $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                    }
                }
            }

            $initialBalance = $initialDebit - $initialCredit;
            $runningBalance = $initialBalance;

            // Filter transfers within the date range and process transactions
            $transfersInRange = $tree->transfers->filter(function ($transfer) use ($request) {
                $withinDateFrom = empty($request['date_from']) || $transfer->date >= $request['date_from'];
                $withinDateTo = empty($request['date_to']) || $transfer->date <= $request['date_to'];

                return $withinDateFrom && $withinDateTo;
            });

            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;

            foreach ($transfersInRange as $transfer) {
                $debit = $account['currency_id'] != null ? $transfer->currency_debit ?? 0 : $transfer->debit ?? 0;
                $credit = $account['currency_id'] != null ? $transfer->currency_creditor ?? 0 : $transfer->creditor ?? 0;
                $transferBalance = $debit - $credit;

                // Update running balance
                $runningBalance += $transferBalance;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'cost_center_name' => $transfer->costCenter->name ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'tree_account_title' => $transfer->treeAccounting->title ?? null,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'debit' => $debit,
                    'credit' => $credit,
                    'transfer_balance' => $transferBalance,
                    'running_balance' => $runningBalance,
                    'rate' => $transfer->currency_transfer_rate,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'tree_account_id' => $tree->id,
                'tree_account_title' => $tree->title,
                'tree_account_serial_number' => $tree->serial_number,
                'initial_debit' => $initialDebit,
                'initial_credit' => $initialCredit,
                'initial_balance' => $initialBalance,
                'transactions' => $transactions,
                'period_debit' => $totalDebit,
                'period_credit' => $totalCredit,
                'period_balance' => $totalDebit - $totalCredit,
                'final_balance' => $initialBalance + ($totalDebit - $totalCredit),
                'show_by_default' => $account['currency_id'] == null ? true : false,

            ];
        });
    }

    public function costCenterReport(array $request)
    {
        $reports = CostCenter::whereIn('id', $request['cost_center_ids'] ?? [])
            ->with([
                'transfers' => function ($query) use ($request) {
                    $query->when($request['date_from'] ?? null, function ($query) use ($request) {
                        return $query->where('date', '>=', $request['date_from']);
                    })
                        ->when($request['date_to'] ?? null, function ($query) use ($request) {
                            return $query->where('date', '<=', $request['date_to']);
                        })
                        ->when(! empty($request['company_ids'] ?? []), function ($query) use ($request) {
                            $query->whereHas('constraint', function ($subQuery) use ($request) {
                                $subQuery->whereIn('company_id', $request['company_ids'])
                                    ->where('active', $request['active']);
                            });
                        });
                },
                'transfers.constraint',
                'transfers.treeAccounting',
                'transfers.currency.currentTranslation',
            ])
            ->get();

        return $reports->map(function ($reports) {
            $transactions = [];
            $totalDebit = 0;
            $totalCredit = 0;

            foreach ($reports->transfers as $transfer) {
                $debit = $transfer->debit ?? 0;
                $credit = $transfer->creditor ?? 0;
                $balance = $debit - $credit;

                $transactions[] = [
                    'transfer_id' => $transfer->id,
                    'tree_account_title' => $transfer->treeAccounting->title,
                    'tree_account_serial_number_dight' => $transfer->treeAccounting->serial_number_dight ?? null,
                    'constraint_id' => $transfer->constraint->id ?? null,
                    'constraint_name' => $transfer->constraint->name ?? null,
                    'constraint_description' => $transfer->constraint->description ?? null,
                    'date' => $transfer->date,
                    'type' => $transfer->constraint->capture_exchange,
                    'number_doc' => $transfer->constraint->number_doc ?? null,
                    'currency_name' => $transfer->currency->name ?? null,
                    'currency_code' => $transfer->currency->code ?? null,
                    'currency_symbol' => $transfer->currency->symbol ?? null,
                    'transfer_name' => $transfer->name,
                    'debit' => $debit,
                    'credit' => $credit,
                    'balance' => $balance,
                ];

                $totalDebit += $debit;
                $totalCredit += $credit;
            }

            return [
                'cost_center_id' => $reports->id,
                'cost_center_name' => $reports->name,
                'transactions' => $transactions,
                'total_debit' => $totalDebit,
                'total_credit' => $totalCredit,
                'total_balance' => $totalDebit - $totalCredit,
            ];
        });
    }
}
