<?php

namespace App\sys\Repository\General;

use App\Models\General\Nationality;

class NationalityRepository
{
    private $columns = [
        'id' => 'id',
        'name' => 'name',
        'active' => 'is_active',
        'created_at' => 'created_at',
        'updated_at' => 'updated_at',
    ];

    // ممكن نعمل فيها موضوع cloumes بتاع الابديت نفس الفكره
    public function getPaginated()
    {
        $column = request('sort_by', null);
        $order = request('sort_order', 'asc');
        $name = request('name', null);
        $limit = request('limit', 15);

        return Nationality::when($name, function ($q, $name) {
            $q->where('name', 'LIKE', "%$name%");
        })->when($column && array_key_exists($column, $this->columns), function ($query) use ($column, $order) {
            if ($column == 'name') {
                $query->leftJoin('nationality_translation', function ($join) {
                    $join->on('nationalities.id', '=', 'nationality_translation.nationality_id')
                        ->where('nationality_translation.lang_id', app('lang_id'));
                })->orderBy('nationality_translation.name', $order)->select('nationalities.*');
            } else {
                $query->orderBy($this->columns[$column], $order);
            }
        })->with('currentTranslation')->paginate($limit);
    }

    public function findByIdOrFail(int $id)
    {
        return Nationality::find($id);
    }

    public function create(array $data)
    {
        $nationality = new Nationality;
        $nationality->name = $data['name'];
        $nationality->is_active = $data['active'];
        $nationality->save();

        return $nationality;
    }

    public function update(Nationality $nationality, array $data)
    {
        $nationality->name = $data['name'] ?? $nationality->name;
        $nationality->is_active = $data['active'] ?? $nationality->is_active;
        $nationality->save();

        return $nationality;
    }

    public function getActive()
    {
        return Nationality::with('currentTranslation')->where('is_active', 1)->get();
    }

    public function getByIdWithTranslation($id)
    {
        return Nationality::with('translations')->find($id);
    }

    public function del(array $ids)
    {
        return Nationality::whereIn('id', $ids)->delete();
    }

    public function getAllChanges($nationality)
    {
        return $nationality->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }
}
