<?php

namespace App\sys\Repository\Invoice;

use App\Models\General\City;
use App\Models\General\Companies;
use App\Models\General\Currency;
use App\Models\General\GuideLanguage;
use App\Models\General\Nationality;
use App\Models\General\OperationData;
use App\Models\General\Service as GeneralService;
use App\Models\invoice\InvoiceServices;
use App\Models\Suppliers\Suppliers;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class InvoiceServicesRepository
{
    public function findByIdOrFail(int $id)
    {
        return InvoiceServices::with($this->relations())->find($id);
    }

    public function findByModel(string $modelType, int $modelId)
    {
        return InvoiceServices::where('model_type', $modelType)
            ->where('model_id', $modelId)
            ->first();
    }

    public function create(array $data)
    {
        $item = new InvoiceServices;

        $item->travel_tourism_type = $data['travel_tourism_type'] ?? 'accommodation';
        $item->type = $data['type'] ?? 'credit';
        $item->travel_tourism_steps = $data['travel_tourism_steps'] ?? 0;
        $item->travel_tourism_done = $data['travel_tourism_done'] ?? 0;
        $item->service_id = $data['service_id'] ?? null;
        $item->domain_id = $data['domain_id'] ?? null;
        $item->invoice_id = $data['invoice_id'] ?? null;
        $item->supplier_id = $data['supplier_id'] ?? null;
        $item->pay_type_id = $data['pay_type_id'] ?? null;
        $item->pay_type = $data['pay_type'] ?? null;
        $item->reserve_number = $data['reserve_number'] ?? null;
        $item->vatt_amount = $data['vatt_amount'] ?? null;
        $item->vatt_Supplier = $data['vatt_Supplier'] ?? null;
        $item->profit = $data['profit'] ?? null;
        $item->employee_tax = $data['employee_tax'] ?? null;
        $item->last_profit = $data['last_profit'] ?? null;
        $item->taxtree = $data['taxtree'] ?? null;
        $item->supplirTree = $data['supplirTree'] ?? null;
        $item->serviceTree = $data['serviceTree'] ?? null;
        $item->payTree = $data['payTree'] ?? null;
        $item->serviceTreeIncom = $data['serviceTreeIncom'] ?? null;
        $item->serviceTypeAndPayType = $data['serviceTypeAndPayType'] ?? null;
        $item->paid_rice = $data['paid_rice'] ?? null;
        $item->supplier_currency_id = $data['supplier_currency_id'] ?? null;
        $item->supplier_currency_transfer_rate = $data['supplier_currency_transfer_rate'] ?? null;
        $item->client_currency_id = $data['client_currency_id'] ?? null;
        $item->client_currency_transfer_rate = $data['client_currency_transfer_rate'] ?? null;
        $item->extra_service_price = $data['extra_service_price'] ?? 0;
        $item->extra_service_tree = $data['extra_service_tree'] ?? null;
        // Attractions/Temples and similar dynamic fields
        $item->daily_program_id = $data['daily_program_id'] ?? null;
        $item->nationality_id = $data['nationality_id'] ?? null;
        $item->adults_count = $data['adults_count'] ?? 0;
        $item->children_count = $data['children_count'] ?? 0;
        $item->adult_price = $data['adult_price'] ?? 0;
        $item->child_price = $data['child_price'] ?? 0;
        $item->adult_total = $data['adult_total'] ?? 0;
        $item->child_total = $data['child_total'] ?? 0;
        // Common dynamic fields across types
        $item->people_count = $data['people_count'] ?? 0;
        $item->daily_cost = $data['daily_cost'] ?? 0;
        $item->extra_cost = $data['extra_cost'] ?? 0;
        $item->total_tax = $data['total_tax'] ?? 0;
        $item->tip_amount = $data['tip_amount'] ?? 0;
        $item->total_tips = $data['total_tips'] ?? 0;
        $item->company_id = $data['company_id'] ?? null;
        $item->city_id = $data['city_id'] ?? null;
        $item->city_to_id = $data['city_to_id'] ?? null;
        $item->operation_data_id = $data['operation_data_id'] ?? null;
        $item->guide_language_id = $data['guide_language_id'] ?? null;
        // tour_guide specifics
        $item->guide_id = $data['guide_id'] ?? null;
        $item->guidance_place = $data['guidance_place'] ?? null;
        $item->grand_total = $data['grand_total'] ?? 0;
        $item->execution_date = $data['execution_date'] ?? null;
        $item->executive_id = $data['executive_id'] ?? null;
        $item->currency_id = $data['currency_id'] ?? null;
        $item->currency_rate = $data['currency_rate'] ?? 0;
        $item->notes = $data['notes'] ?? $item->notes ?? null;
        $item->purchase_price = $data['purchase_price'] ?? null;
        $item->purchase_commission = $data['purchase_commission'] ?? null;
        $item->purchase_rate = $data['purchase_rate'] ?? null;
        $item->purchase_tax_id = $data['purchase_tax_id'] ?? null;
        $item->purchase_tax_amount = $data['purchase_tax_amount'] ?? null;
        $item->purchase_currency_id = $data['purchase_currency_id'] ?? null;
        $item->purchase_tax_rate = $data['purchase_tax_rate'] ?? null;
        $item->sale_price = $data['sale_price'] ?? null;
        $item->sale_commission = $data['sale_commission'] ?? null;
        $item->sale_rate = $data['sale_rate'] ?? null;
        $item->sale_tax_id = $data['sale_tax_id'] ?? null;
        $item->sale_tax_amount = $data['sale_tax_amount'] ?? null;
        $item->sale_currency_id = $data['sale_currency_id'] ?? null;
        $item->sale_tax_rate = $data['sale_tax_rate'] ?? null;
        $item->model_type = $data['model_type'] ?? null;
        $item->model_id = $data['model_id'] ?? null;
        $item->profile_id = $data['profile_id'] ?? null;
        $item->reservation_id = $data['reservation_id'] ?? null;
        $item->reservation_model = $data['reservation_model'] ?? null;

        $item->save();

        return $item->load($this->relations());
    }

    public function update(InvoiceServices $item, array $data)
    {
        $item->travel_tourism_type = $data['travel_tourism_type'] ?? $item->travel_tourism_type;
        $item->travel_tourism_steps = $data['travel_tourism_steps'] ?? $item->travel_tourism_steps;
        $item->travel_tourism_done = $data['travel_tourism_done'] ?? $item->travel_tourism_done;
        $item->service_id = $data['service_id'] ?? $item->service_id;
        $item->domain_id = $data['domain_id'] ?? $item->domain_id;
        $item->invoice_id = $data['invoice_id'] ?? $item->invoice_id;
        $item->supplier_id = $data['supplier_id'] ?? $item->supplier_id;
        $item->pay_type_id = $data['pay_type_id'] ?? $item->pay_type_id;
        $item->pay_type = $data['pay_type'] ?? $item->pay_type;
        $item->reserve_number = $data['reserve_number'] ?? $item->reserve_number;
        $item->vatt_amount = $data['vatt_amount'] ?? $item->vatt_amount;
        $item->vatt_Supplier = $data['vatt_Supplier'] ?? $item->vatt_Supplier;
        $item->profit = $data['profit'] ?? $item->profit;
        $item->employee_tax = $data['employee_tax'] ?? $item->employee_tax;
        $item->last_profit = $data['last_profit'] ?? $item->last_profit;
        $item->taxtree = $data['taxtree'] ?? $item->taxtree;
        $item->supplirTree = $data['supplirTree'] ?? $item->supplirTree;
        $item->serviceTree = $data['serviceTree'] ?? $item->serviceTree;
        $item->payTree = $data['payTree'] ?? $item->payTree;
        $item->serviceTreeIncom = $data['serviceTreeIncom'] ?? $item->serviceTreeIncom;
        $item->serviceTypeAndPayType = $data['serviceTypeAndPayType'] ?? $item->serviceTypeAndPayType;
        $item->paid_rice = $data['paid_rice'] ?? $item->paid_rice;
        $item->supplier_currency_id = $data['supplier_currency_id'] ?? $item->supplier_currency_id;
        $item->supplier_currency_transfer_rate = $data['supplier_currency_transfer_rate'] ?? $item->supplier_currency_transfer_rate;
        $item->client_currency_id = $data['client_currency_id'] ?? $item->client_currency_id;
        $item->client_currency_transfer_rate = $data['client_currency_transfer_rate'] ?? $item->client_currency_transfer_rate;
        $item->extra_service_price = $data['extra_service_price'] ?? $item->extra_service_price;
        $item->extra_service_tree = $data['extra_service_tree'] ?? $item->extra_service_tree;
        $item->created = $data['created'] ?? $item->created;
        $item->modified = $data['modified'] ?? $item->modified;
        $item->purchase_price = $data['purchase_price'] ?? $item->purchase_price;
        $item->purchase_commission = $data['purchase_commission'] ?? $item->purchase_commission;
        $item->purchase_rate = $data['purchase_rate'] ?? $item->purchase_rate;
        $item->purchase_tax_id = $data['purchase_tax_id'] ?? $item->purchase_tax_id;
        $item->purchase_tax_amount = $data['purchase_tax_amount'] ?? $item->purchase_tax_amount;
        $item->purchase_currency_id = $data['purchase_currency_id'] ?? $item->purchase_currency_id;
        $item->purchase_tax_rate = $data['purchase_tax_rate'] ?? $item->purchase_tax_rate;
        $item->sale_price = $data['sale_price'] ?? $item->sale_price;
        $item->sale_commission = $data['sale_commission'] ?? $item->sale_commission;
        $item->sale_rate = $data['sale_rate'] ?? $item->sale_rate;
        $item->sale_tax_id = $data['sale_tax_id'] ?? $item->sale_tax_id;
        $item->sale_tax_amount = $data['sale_tax_amount'] ?? $item->sale_tax_amount;
        $item->sale_currency_id = $data['sale_currency_id'] ?? $item->sale_currency_id;
        $item->sale_tax_rate = $data['sale_tax_rate'] ?? $item->sale_tax_rate;
        // Dynamic type fields
        $item->daily_program_id = $data['daily_program_id'] ?? $item->daily_program_id;
        $item->nationality_id = $data['nationality_id'] ?? $item->nationality_id;
        $item->adults_count = $data['adults_count'] ?? $item->adults_count;
        $item->children_count = $data['children_count'] ?? $item->children_count;
        $item->adult_price = $data['adult_price'] ?? $item->adult_price;
        $item->child_price = $data['child_price'] ?? $item->child_price;
        $item->adult_total = $data['adult_total'] ?? $item->adult_total;
        $item->child_total = $data['child_total'] ?? $item->child_total;
        // Common dynamic fields across types
        $item->people_count = $data['people_count'] ?? $item->people_count;
        $item->daily_cost = $data['daily_cost'] ?? $item->daily_cost;
        $item->extra_cost = array_key_exists('extra_cost', $data) ? $data['extra_cost'] : $item->extra_cost;
        $item->total_tax = array_key_exists('total_tax', $data) ? $data['total_tax'] : $item->total_tax;
        $item->tip_amount = $data['tip_amount'] ?? $item->tip_amount;
        $item->total_tips = $data['total_tips'] ?? $item->total_tips;
        $item->company_id = $data['company_id'] ?? $item->company_id;
        $item->city_id = $data['city_id'] ?? $item->city_id;
        $item->city_to_id = $data['city_to_id'] ?? $item->city_to_id;
        $item->operation_data_id = $data['operation_data_id'] ?? $item->operation_data_id;
        $item->guide_language_id = $data['guide_language_id'] ?? $item->guide_language_id;
        // tour_guide specifics
        $item->guide_id = array_key_exists('guide_id', $data) ? $data['guide_id'] : $item->guide_id;
        $item->guidance_place = array_key_exists('guidance_place', $data) ? $data['guidance_place'] : $item->guidance_place;
        $item->grand_total = $data['grand_total'] ?? $item->grand_total;
        $item->execution_date = array_key_exists('execution_date', $data) ? $data['execution_date'] : $item->execution_date;
        $item->executive_id = $data['executive_id'] ?? $item->executive_id;
        $item->currency_id = $data['currency_id'] ?? $item->currency_id;
        $item->currency_rate = $data['currency_rate'] ?? $item->currency_rate;
        $item->notes = array_key_exists('notes', $data) ? $data['notes'] : $item->notes;
        $item->type = $data['type'] ?? $item->type;
        $item->model_type = $data['model_type'] ?? $item->model_type;
        $item->model_id = $data['model_id'] ?? $item->model_id;
        $item->profile_id = $data['profile_id'] ?? $item->profile_id;
        $item->reservation_id = $data['reservation_id'] ?? $item->reservation_id;
        $item->reservation_model = $data['reservation_model'] ?? $item->reservation_model;

        $item->save();

        return $item->load($this->relations());
    }

    public function delete(InvoiceServices $item)
    {
        return $item->delete();
    }

    public function deleteManyWithRules(array $ids): array
    {
        $items = InvoiceServices::whereIn('id', $ids)->get();

        $deletable = [];
        $blocked = [];

        foreach ($items as $item) {
            $steps = (int)($item->travel_tourism_steps ?? 0);
            $done = (int)($item->travel_tourism_done ?? 0);
            // block delete if any of them equals 1 → allow only when both are not 1
            if ($steps != 1 && $done != 1) {
                $deletable[] = $item->id;
            } else {
                $blocked[] = [
                    'id' => $item->id,
                    'reason' => 'cannot delete: travel_tourism_steps = 1 or travel_tourism_done = 1',
                ];
            }
        }

        if (!empty($deletable)) {
            InvoiceServices::whereIn('id', $deletable)->delete();
        }

        return [
            'deleted_ids' => $deletable,
            'blocked' => $blocked,
        ];
    }

    public function getByProfile(int $profileId, string $serviceType)
    {
        $limit = request('limit', 15);

        return InvoiceServices::with($this->relations())
            ->where([['profile_id', $profileId], ['travel_tourism_type', $serviceType]])->paginate($limit);
    }

    public function getByDaily(int $daily, string $serviceType)
    {
        $limit = request('limit', 15);

        return InvoiceServices::with($this->relations())
            ->where([['daily_program_id', $daily], ['travel_tourism_type', $serviceType]])->paginate($limit);
    }

    public function getList(array $filters = [])
    {
        $query = InvoiceServices::with($this->relations())->query();

        if (isset($filters['profile_id'])) {
            $query->where('profile_id', $filters['profile_id']);
        }

        if (isset($filters['travel_tourism_type'])) {
            $query->where('travel_tourism_type', $filters['travel_tourism_type']);
        }

        if (isset($filters['service_id'])) {
            $query->where('service_id', $filters['service_id']);
        }

        if (isset($filters['invoice_id'])) {
            $query->where('invoice_id', $filters['invoice_id']);
        }

        // Add pagination if needed
        $limit = $filters['limit'] ?? 15;

        return $query->paginate($limit);
    }

    private function getFillableFields(): array
    {
        return [
            'travel_tourism_type',
            'travel_tourism_steps',
            'travel_tourism_done',
            'service_id',
            'domain_id',
            'invoice_id',
            'supplier_id',
            'pay_type_id',
            'pay_type',
            'reserve_number',
            'vatt_amount',
            'vatt_Supplier',
            'profit',
            'employee_tax',
            'last_profit',
            'taxtree',
            'supplirTree',
            'serviceTree',
            'payTree',
            'serviceTreeIncom',
            'serviceTypeAndPayType',
            'paid_rice',
            'supplier_currency_id',
            'supplier_currency_transfer_rate',
            'client_currency_id',
            'client_currency_transfer_rate',
            'extra_service_price',
            'extra_service_tree',
            'created',
            'modified',
            'purchase_price',
            'purchase_commission',
            'purchase_rate',
            'purchase_tax_id',
            'purchase_tax_amount',
            'purchase_currency_id',
            'purchase_tax_rate',
            'sale_price',
            'sale_commission',
            'sale_rate',
            'sale_tax_id',
            'sale_tax_amount',
            'sale_currency_id',
            'sale_tax_rate',
            'type',
            'model_type',
            'model_id',
            'profile_id',
        ];
    }

    private function relations(): array
    {
        return [
            'dailyProgram',
            'nationality.currentTranslation',
            'executive',
            'currency.currentTranslation',
            'service.currentTranslation',
            'supplier.currentTranslation',
            'profile',
            'guide',
            'guideLanguage.currentTranslation',
            'taxes',
            // Added for daily_transportation
            'company.currentTranslation',
            'city.currentTranslation',
            'cityTo.currentTranslation',
            'operationData.currentTranslation',
            // Added for accommodation search
            'reservation.accommodation.currentTranslation',
            'reservation.rooms.roomType',
            'model.roomType',
            'model.mealPlan',
            'model.roomView'
        ];
    }

    public function getAllChanges($invoiceService)
    {
        return $invoiceService->audits()
            ->with(['user' => function ($query) {
                $query->select('id', 'name');
            }])
            ->get()
            ->map(function ($audit) {
                $oldValues = $audit->old_values ?? [];
                $newValues = $audit->new_values ?? [];

                // Collect related IDs from old/new values
                $companyIds = array_values(array_unique(array_filter([
                    $oldValues['company_id'] ?? null,
                    $newValues['company_id'] ?? null,
                ])));
                $supplierIds = array_values(array_unique(array_filter([
                    $oldValues['supplier_id'] ?? null,
                    $newValues['supplier_id'] ?? null,
                ])));
                $serviceIds = array_values(array_unique(array_filter([
                    $oldValues['service_id'] ?? null,
                    $newValues['service_id'] ?? null,
                ])));
                $currencyIds = array_values(array_unique(array_filter([
                    $oldValues['currency_id'] ?? null,
                    $newValues['currency_id'] ?? null,
                    $oldValues['purchase_currency_id'] ?? null,
                    $newValues['purchase_currency_id'] ?? null,
                    $oldValues['sale_currency_id'] ?? null,
                    $newValues['sale_currency_id'] ?? null,
                ])));
                $userIds = array_values(array_unique(array_filter([
                    $oldValues['executive_id'] ?? null,
                    $newValues['executive_id'] ?? null,
                    $oldValues['guide_id'] ?? null,
                    $newValues['guide_id'] ?? null,
                    $audit->user_id ?? null,
                ])));
                $nationalityIds = array_values(array_unique(array_filter([
                    $oldValues['nationality_id'] ?? null,
                    $newValues['nationality_id'] ?? null,
                ])));
                $guideLanguageIds = array_values(array_unique(array_filter([
                    $oldValues['guide_language_id'] ?? null,
                    $newValues['guide_language_id'] ?? null,
                ])));
                $cityIds = array_values(array_unique(array_filter([
                    $oldValues['city_id'] ?? null,
                    $newValues['city_id'] ?? null,
                ])));
                $operationDataIds = array_values(array_unique(array_filter([
                    $oldValues['operation_data_id'] ?? null,
                    $newValues['operation_data_id'] ?? null,
                ])));

                // Fetch names/maps
                $companies = !empty($companyIds) ? Companies::whereIn('id', $companyIds)->pluck('name_company', 'id')->toArray() : [];
                $suppliers = !empty($supplierIds) ? Suppliers::whereIn('id', $supplierIds)->pluck('name', 'id')->toArray() : [];
                $services = !empty($serviceIds) ? GeneralService::whereIn('id', $serviceIds)->pluck('title', 'id')->toArray() : [];
                $currencies = !empty($currencyIds) ? Currency::whereIn('id', $currencyIds)->pluck('code', 'id')->toArray() : [];
                $users = !empty($userIds) ? User::whereIn('id', $userIds)->pluck('name', 'id')->toArray() : [];
                $nationalities = !empty($nationalityIds) ? Nationality::whereIn('id', $nationalityIds)->pluck('name', 'id')->toArray() : [];
                $guideLanguages = !empty($guideLanguageIds) ? GuideLanguage::whereIn('id', $guideLanguageIds)->pluck('title', 'id')->toArray() : [];
                $cities = !empty($cityIds) ? City::whereIn('id', $cityIds)->pluck('name', 'id')->toArray() : [];
                $operations = !empty($operationDataIds) ? OperationData::whereIn('id', $operationDataIds)->pluck('title', 'id')->toArray() : [];

                // Enrich readable names
                foreach (['company_id' => $companies, 'supplier_id' => $suppliers, 'service_id' => $services, 'currency_id' => $currencies, 'purchase_currency_id' => $currencies, 'sale_currency_id' => $currencies, 'executive_id' => $users, 'guide_id' => $users, 'nationality_id' => $nationalities, 'guide_language_id' => $guideLanguages, 'city_id' => $cities, 'operation_data_id' => $operations] as $key => $map) {
                    if (isset($oldValues[$key]) && isset($map[$oldValues[$key]])) {
                        $oldValues[$key . '_name'] = $map[$oldValues[$key]];
                    }
                    if (isset($newValues[$key]) && isset($map[$newValues[$key]])) {
                        $newValues[$key . '_name'] = $map[$newValues[$key]];
                    }
                }

                return [
                    'audit_id' => $audit->id,
                    'user_id' => $audit->user_id ?? null,
                    'user' => $audit->user ? $audit->user->toArray() : null,
                    'old_values' => $oldValues,
                    'new_values' => $newValues,
                    'changed_at' => $audit->created_at,
                    'event' => $audit->event,
                    'ip_address' => $audit->ip_address,
                    'user_agent' => $audit->user_agent,
                ];
            })
            ->values();
    }

    public function summeryinCurrency($profile, $currency)
    {
        /*
           return InvoiceServices::select(
               DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id) as final_currency_id'),
               DB::raw('MIN(currencies.name) as currency_name'),
               DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
               DB::raw('SUM(invoice_services.	grand_total) as 	grand_total'))
               ->join('currencies', 'currencies.id', '=', DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->where('invoice_services.profile_id', $profile)
               ->whereIn(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'), $currency)
               ->groupBy(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->groupBy('invoice_services.currency_id', 'currencies.name')
               ->get();
        */
        return InvoiceServices::select(
            'invoice_services.currency_id as final_currency_id',
            DB::raw('MIN(currencies.name) as currency_name'),
            DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
            DB::raw('SUM(invoice_services.grand_total) as grand_total')
        )
            ->join('currencies', 'currencies.id', '=', 'invoice_services.currency_id')
            ->where('invoice_services.profile_id', $profile)
            ->whereIn('invoice_services.currency_id', $currency)
            ->groupBy('invoice_services.currency_id')
            ->get();

    }

    public function summeryinCurrencyNotIn($profile, $currency)
    {

        /*
           return InvoiceServices::select(
               DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id) as final_currency_id'),
               DB::raw('MIN(currencies.name) as currency_name'),
               DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
               DB::raw('SUM(invoice_services.	grand_total) as 	grand_total'))
               ->join('currencies', 'currencies.id', '=', DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->where('invoice_services.profile_id', $profile)
               ->whereNotIn(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'), $currency)
               ->groupBy(DB::raw('COALESCE(invoice_services.currency_id, invoice_services.purchase_currency_id)'))
               ->groupBy('invoice_services.currency_id', 'currencies.name')
               ->get();
        */
        return InvoiceServices::select(
            'invoice_services.currency_id as final_currency_id',
            DB::raw('MIN(currencies.name) as currency_name'),
            DB::raw('SUM(invoice_services.purchase_price) as purchase_price'),
            DB::raw('SUM(invoice_services.grand_total) as grand_total')
        )
            ->join('currencies', 'currencies.id', '=', 'invoice_services.currency_id')
            ->where('invoice_services.profile_id', $profile)
            ->whereNotIn('invoice_services.currency_id', $currency)
            ->groupBy('invoice_services.currency_id')
            ->get();

    }

    public function CheckInvoiceServices($services, $profile)
    {
        return InvoiceServices::whereIn('travel_tourism_type', $services)->where('profile_id', $profile)->exists();
    }

    public function getServicesForProfile($profile)
    {
        $services = ['tour_guide', 'tour_reps', 'daily_transportation', 'dining_entertainment', 'daily_program_id'];

        return InvoiceServices::with('dailyProgram', 'service', 'guideLanguage')
            ->whereIn('travel_tourism_type', $services)
            ->where('profile_id', $profile)
            ->orderBy('daily_program_id', 'asc')
            ->get()
            ->groupBy('daily_program_id')
            ->map(function ($group) {
                $dailyProgram = $group->first()->dailyProgram;

                return [
                    'day_number' => $dailyProgram->day_number ?? null,
                    'day_date' => $dailyProgram->day_date ?? null,
                    'services' => $group->map(function ($item) {
                        return [
                            'title' => $item->service->title ?? null,
                            'type' => $item->service->type ?? null,
                            'notes' => $item->notes ?? null,
                        ];
                    })->values(),
                ];
            })->values();
    }

    public function getOtherSerForProfile($services, $profile)
    {
        //   return InvoiceServices::with('service','city','cityTo')->where('travel_tourism_type', $services)->where('profile_id',$profile)->get();

        return InvoiceServices::with([
            'service:id,title',
            'city:id,name',
            'cityTo:id,name',
        ])
            ->where('travel_tourism_type', $services)
            ->where('profile_id', $profile)
            ->get(['id', 'service_id', 'city_id', 'city_to_id', 'people_count'])
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'service_title' => $item->service->title ?? null,
                    'city_name' => $item->city->name ?? null,
                    'city_to_name' => $item->cityTo->name ?? null,
                    'people_count' => $item->people_count,
                ];
            });
    }

    public function search()
    {
        // Define all filter variables first
        $type = request('type', null);
        // Handle type as string, array, or comma-separated string
        if ($type) {
            if (is_string($type)) {
                $type = strpos($type, ',') !== false ? explode(',', $type) : [$type];
            } elseif (!is_array($type)) {
                $type = [$type];
            }
        }
        $supplier_name = request('supplier_name', null);
        $profile_nam = request('profile_num', null);
        $currency = request('currency_id', null);
        $accommodation_reservation_status = request('accommodation_reservation_status', null);
        $arrivalDateFrom = request('arrival_date_from', null);
        $arrivalDateTo = request('arrival_date_to', null);
        $checkin = request('checkin', null); // الهوتيل
        $checkout = request('checkout', null);
        $execution_date = request('execution_date', null); // خدمات الاخري
        $daily_date = request('daily_date', null); //services
        $supplier_id = request('supplier_id', null);
        $serviceName = request('service_name', null);
        $cityFrom = request('city_from', null);
        $cityTo = request('city_to', null);
        $guideLanguage = request('guide_language', null);
        $hotel_name = request('accommodation_name', null);
        $limit = request('limit', 20);
        $room_type = request('room_type', null);
        $room_view = request('room_view', null);
        $meal_type = request('meal_type', null);

        $query = InvoiceServices::with($this->relations());

        // 1. Accommodation type filter (hotel/cruise) through reservation relationship
        $query->when($type, function ($q) use ($type) {
            // Convert to array if it's a string
            $types = is_array($type) ? $type : [$type];

            $reservationTypes = array_intersect($types, ['hotel', 'cruise']);
            $otherTypes = array_diff($types, ['hotel', 'cruise']);

            // If we have both accommodation types and other types, use OR condition
            if (!empty($reservationTypes) && !empty($otherTypes)) {
                return $q->where(function ($subQ) use ($reservationTypes, $otherTypes) {
                    // Search in reservations for hotel/cruise
                    $subQ->whereHas('reservation', function ($resQ) use ($reservationTypes) {
                        $resQ->whereHas('accommodation', function ($accQ) use ($reservationTypes) {
                            $accQ->whereIn('type', $reservationTypes);
                        });
                    })
                    // OR search in travel_tourism_type for other types
                    ->orWhereIn('travel_tourism_type', $otherTypes);
                });
            }
            // If only accommodation types
            elseif (!empty($reservationTypes)) {
                return $q->whereHas('reservation', function ($subQ) use ($reservationTypes) {
                    return $subQ->whereHas('accommodation', function ($subQ) use ($reservationTypes) {
                        $subQ->whereIn('type', $reservationTypes);
                    });
                });
            }
            // If only other types
            else {
                return $q->whereIn('travel_tourism_type', $otherTypes);
            }
        });

        $query->when($hotel_name, function ($q) use ($hotel_name) {
            return $q->whereHas('reservation', function ($subQ) use ($hotel_name) {
                $subQ->whereHas('accommodation', function ($subQ) use ($hotel_name) {
                    $subQ->where('name', 'like', "%$hotel_name%");
                });
            });
        });
        $query->when($accommodation_reservation_status, function ($q) use ($accommodation_reservation_status) {
            return $q->whereHas('reservation', function ($subQ) use ($accommodation_reservation_status) {
                $subQ->where('status', $accommodation_reservation_status);
            });
        });


        // 2. Profile ID filter
        $query->when($currency, function ($q) use ($currency) {
            return $q->where('currency_id', $currency);
        });

        $query->when($supplier_id, function ($q) use ($supplier_id) {
            return $q->where('supplier_id', $supplier_id);
        });

        $query->when($supplier_name, function ($q) use ($supplier_name) {
            return $q->whereHas('supplier', function ($subQ) use ($supplier_name) {
                $subQ->where('supplier_name', 'like', "%$supplier_name%");
            });
        });

        // 3. Profile arrival and departure dates

        $query->when($arrivalDateFrom, function ($q) use ($arrivalDateFrom) {
            return $q->whereHas('profile', function ($subQ) use ($arrivalDateFrom) {
                $subQ->where('arrival_date', '<=', $arrivalDateFrom);
            });
        });

        $query->when($arrivalDateTo, function ($q) use ($arrivalDateTo) {
            return $q->whereHas('profile', function ($subQ) use ($arrivalDateTo) {
                $subQ->where('departure_date', '>=', $arrivalDateTo);
            });
        });


        $query->when($profile_nam, function ($q) use ($profile_nam) {
            return $q->whereHas('profile', function ($subQ) use ($profile_nam) {
                $subQ->where('profile_number', 'like', "%$profile_nam%");
            });
        });

        // 4. Service name filter
        $query->when($serviceName, function ($q) use ($serviceName) {
            return $q->whereHas('service', function ($subQ) use ($serviceName) {
                $subQ->where('title', 'like', '%' . $serviceName . '%');
            });
        });

        // 5. City filters (from and to)
        $query->when($cityFrom, function ($q) use ($cityFrom) {
            return $q->where('city_id', $cityFrom);
        });

        $query->when($cityTo, function ($q) use ($cityTo) {
            return $q->where('city_to_id', $cityTo);
        });
        $query->when($execution_date, function ($q) use ($execution_date) {
            return $q->where('execution_date', $execution_date);
        });
        $query->when($daily_date, function ($q) use ($daily_date) {
            return $q->whereHas('dailyProgram', function ($subQ) use ($daily_date) {
                $subQ->where('day_date', $daily_date);
            });
        });

        $query->when($checkin, function ($q) use ($checkin) {
            return $q->whereHas('model', function ($subQ) use ($checkin) {
                $subQ->where('check_in', $checkin);
            });
        });
        $query->when($checkout, function ($q) use ($checkout) {
            return $q->whereHas('model', function ($subQ) use ($checkout) {
                $subQ->where('check_out', $checkout);
            });
        });

        $query->when($room_type, function ($q) use ($room_type) {
            return $q->whereHas('model', function ($subQ) use ($room_type) {
                $subQ->where('room_type_id', $room_type);
            });
        });
        $query->when($room_view, function ($q) use ($room_view) {
            return $q->whereHas('model', function ($subQ) use ($room_view) {
                $subQ->where('room_view_id', $room_view);
            });
        });
        $query->when($meal_type, function ($q) use ($meal_type) {
            return $q->whereHas('model', function ($subQ) use ($meal_type) {
                $subQ->where('meal_plan_id', $meal_type);
            });
        });
        // 6. Guide language filter
        $query->when($guideLanguage, function ($q) use ($guideLanguage) {
            return $q->where('guide_language_id', $guideLanguage);
        });
        // Additional filters

        // Pagination
        $results = $query->paginate($limit);

        // Transform results to include structured data
        $results->getCollection()->transform(function ($item) {
            return [
                'id' => $item->id,
                'type' => $item->travel_tourism_type == "accommodation"?$this->getHotelOrCruiseName($item):$item->travel_tourism_type,
                'service_name' => $item->service->currentTranslation->title ?? null,
                'room_number' =>$item->travel_tourism_type == "accommodation"? $this->getRoomNum($item):null,
                'room_type' => $item->travel_tourism_type == "accommodation"?$this->getRoomType($item):null,
                'room_view' => $item->travel_tourism_type == "accommodation"?$this->getRoomView($item):null,
                'room_meal' =>$item->travel_tourism_type == "accommodation"? $this->getRoomMeal($item):null,
                'accommodation_name' =>$item->travel_tourism_type == "accommodation"? $this->getAccommodationName($item):null,
                'hotel_or_cruise' =>$item->travel_tourism_type == "accommodation"? $this->getHotelOrCruiseName($item):null,
                'profile_id' => $item->profile_id,
                'profile_number' => $item->profile->profile_number ?? null,
                'arrival_date' => $item->profile->arrival_date ?? null,
                'departure_date' => $item->profile->departure_date ?? null,
                'city_from' => $item->city->currentTranslation->name ?? null,
                'city_to' => $item->cityTo->currentTranslation->name ?? null,
                'guide_language' => $item->guideLanguage->currentTranslation->name ?? null,
                'execution_date' => $item->execution_date,
                'grand_total' =>$item->travel_tourism_type == "accommodation"?$item->purchase_price:$item->grand_total,
                'total_paid' =>0,
                'remainder' =>$item->travel_tourism_type == "accommodation"?$item->purchase_price:$item->grand_total,
                'notes' => $item->notes,
                'supplier_name' => $item->supplier->currentTranslation->name ?? null,
                'supplier_id' => $item->supplier_id,
                'reset_num' => 454545,
                'accommodation_confirm_number' => $item->reservation->confirmation_num ?? null,
                "accommodation_status" => $item->reservation->status ?? null,
                'daily_program_id' => $item->daily_program_id,
                'daily_program_day' => $item->dailyProgram->day_date ?? null,
                'currency_id' => $item->currency_id,
                'currency_name' => $item->currency->currentTranslation->name ?? null,
                'executive_id' => $item->executive_id,
                'reservation_id' => $item->reservation_id ?? null,
                'room_id' => $item->model_id ?? null,
                "check_in" => $item->travel_tourism_type == "accommodation"?$item->model->check_in??null:null,
                "check_out" => $item->travel_tourism_type == "accommodation"?$item->model->check_out??null:null,
            ];
        });

        // Return pagination with transformed data
        return [
            'data' => $results->items(),
            'pagination' => [
                'current_page' => $results->currentPage(),
                'last_page' => $results->lastPage(),
                'per_page' => $results->perPage(),
                'total' => $results->total(),
                'from' => $results->firstItem(),
                'to' => $results->lastItem(),
                'has_more_pages' => $results->hasMorePages(),
            ]
        ];
    }

    /**
     * Get room name from accommodation reservation
     */
    private function getRoomNum($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $room = $item->model;
            if ($room) {
                return $room->room_number ?? null;
            }
        }
        return null;
    }
    private function getRoomType($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $room = $item->model;
            if ($room && $room->roomType) {
                return $room->roomType->currentTranslation->name ?? $room->roomType->name ?? null;
            }
        }
        return null;
    }
    private function getRoomView($item)
    {
        if ($item->travel_tourism_type == 'accommodation'  && $item->model) {
            $roomView = $item->model->roomView;
            if ($roomView) {
                return $roomView->currentTranslation->name ?? $roomView->name ??null;
            }
        }
        return null;
    }
    private function getRoomMeal($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->model) {
            $mealPlan = $item->model->mealPlan;
            if ($mealPlan) {
                return $mealPlan->currentTranslation->name ?? $mealPlan->name ?? null;
            }
        }
        return null;
    }

    /**
     * Get accommodation name
     */
    private function getAccommodationName($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->reservation && $item->reservation->accommodation) {
            return $item->reservation->accommodation->currentTranslation->name ?? null;
        }
        return null;
    }

    /**
     * Get hotel or cruise name based on accommodation type
     */
    private function getHotelOrCruiseName($item)
    {
        if ($item->travel_tourism_type == 'accommodation' && $item->reservation) {
            $accommodation = $item->reservation->accommodation;
            if ($accommodation) {
                $type = $item->reservation->accommodation->type; // hotel or cruise
                return $type;
            }
        }
        return null;
    }
}
