<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\PaytypeRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class PaytypeService extends Services
{
    protected PaytypeRepository $paytypeRepository;

    public function __construct(PaytypeRepository $paytypeRepository)
    {
        $this->paytypeRepository = $paytypeRepository;
    }

    public function getPaginatedPaytypes()
    {
        return $this->paytypeRepository->getPaginated();
    }

    public function getByCurrnecy($currnecy)
    {
        return $this->paytypeRepository->getByCurrnecy($currnecy);
    }

    public function getPaytypeById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->paytypeRepository->findByIdOrFail($id);
    }

    public function createPaytype($request)
    {
        if (! $this->validatePaytype($request)) {
            return false;
        }

        $paytype = $this->paytypeRepository->create($request);
        $this->syncTranslations($paytype);

        return $paytype;
    }

    public function updatePaytype($request)
    {
        if (! $this->validUpdated($request)) {
            return false;
        }

        $result = $this->paytypeRepository->update($request);

        if (! $result) {
            $this->setError(['id' => ['Paytype not found']]);

            return false;
        }

        return $result;
    }

    private function validatePaytype($request)
    {
        $rules = [
            'title' => ['required', 'string', 'max:255', 'unique:pay_type,title'],
            'accounting' => ['required', 'array', 'min:1'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            $inputAccounts = $request['accounting'] ?? [];

            if (empty($inputAccounts)) {
                $validator->errors()->add('accounting', 'يجب إرسال محاسبة واحدة على الأقل');

                return;
            }

            $sentKeys = [];
            $missingFieldsErrors = [];

            foreach ($inputAccounts as $index => $item) {
                $required = ['currency_id', 'percentage', 'tree_accounting_transfer_id', 'tree_account_commission_id'];
                $missing = array_diff($required, array_keys($item));

                if ($missing) {
                    $missingFieldsErrors[] = "العنصر رقم {$index} ناقص: ".implode(', ', $missing);

                    continue;
                }

                $sentKeys[] = $item['currency_id'];
            }

            if ($missingFieldsErrors) {
                foreach ($missingFieldsErrors as $err) {
                    $validator->errors()->add('accounting', $err);
                }

                return;
            }

            // Check for duplicate currencies
            if (count($sentKeys) !== count(array_unique($sentKeys))) {
                $validator->errors()->add('accounting', 'لا يمكن تكرار نفس العملة');

                return;
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    private function validUpdated($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
            'title' => ['sometimes', 'string', 'max:255', 'unique:pay_type,title,'.$request['id']],
            'accounting' => ['sometimes', 'array', 'min:1'],
        ];
        $validator = Validator::make($request, $rules);
        $validator->after(function ($validator) use ($request) {
            if (isset($request['accounting'])) {
                $inputAccounts = $request['accounting'] ?? [];

                if (empty($inputAccounts)) {
                    $validator->errors()->add('accounting', 'يجب إرسال محاسبة واحدة على الأقل');

                    return;
                }

                $sentKeys = [];
                $missingFieldsErrors = [];

                foreach ($inputAccounts as $index => $item) {
                    $required = ['currency_id', 'percentage', 'tree_accounting_transfer_id', 'tree_account_commission_id'];
                    $missing = array_diff($required, array_keys($item));

                    if ($missing) {
                        $missingFieldsErrors[] = "العنصر رقم {$index} ناقص: ".implode(', ', $missing);

                        continue;
                    }

                    $sentKeys[] = $item['currency_id'];
                }

                if ($missingFieldsErrors) {
                    foreach ($missingFieldsErrors as $err) {
                        $validator->errors()->add('accounting', $err);
                    }

                    return;
                }

                // Check for duplicate currencies
                if (count($sentKeys) !== count(array_unique($sentKeys))) {
                    $validator->errors()->add('accounting', 'لا يمكن تكرار نفس العملة');

                    return;
                }
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return true;
    }

    public function del(array $ids)
    {
        $result = $this->paytypeRepository->del($ids);

        if (is_array($result) && ! $result['success']) {
            $this->setError(['delete' => [$result['message']]]);

            return false;
        }

        return $result;
    }

    public function getByidWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pay_type,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $paytype = $this->paytypeRepository->getByIdWithTranslation($id);

        if (! $paytype) {
            $this->setError(['id' => ['Paytype not found']]);

            return false;
        }

        return $this->syncTranslations($paytype, true);
    }

    public function getAllChanges($paytype)
    {
        return $this->paytypeRepository->getAllChanges($paytype);
    }
}
