<?php

namespace App\sys\Services\Accounting;

use App\Models\Accounting\AccountsStructure;
use App\sys\Helper;
use App\sys\Repository\Accounting\TreeAccountingRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class TreeAccountingServices extends Services
{
    private $tree;

    public function __construct()
    {
        $this->tree = new TreeAccountingRepository;
    }

    public function gettreeByCurrency($id)
    {
        return $this->tree->getByCurrency($id);
    }

    public function getParents()
    {
        return $this->tree->getParents();
    }

    public function getActive()
    {
        return $this->tree->getActive();
    }

    public function getByParentId(int $parentId)
    {
        return $this->tree->getByParentId($parentId);
    }

    public function getList()
    {
        return $this->tree->getList();
    }

    public function getPaginated()
    {
        return $this->tree->getPaginated();
    }

    public function findById(int $id)
    {
        return $this->tree->findById($id);
    }

    public function updateBasic(array $data)
    {
        $rules = [
            'id' => 'required|integer|exists:tree_accounting,id',
            'title' => 'sometimes|required|string|max:250|unique:tree_accounting,title,'.$data['id'],
            'description' => 'sometimes|nullable|string',
            'active' => 'sometimes|in:0,1',
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->tree->updated($data);

    }

    public function create($data)
    {
        $rules = [
            'parent_id' => 'required|integer',
            'title' => 'required|string|max:250|unique:tree_accounting,title',
            'description' => 'nullable|string',
            'serial_number_dight' => 'required|string|unique:tree_accounting,serial_number_dight',
            'accounts_structure_id' => 'required|integer|exists:accounts_structure,id',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'company_id' => 'required|integer|exists:companies,id',
            'credit' => 'nullable|numeric|min:0',
            'debit' => 'nullable|numeric|min:0',
        ];
        $parent = null;
        $validator = Validator::make($data, $rules);
        $validator->after(function ($v) use ($data, $parent) {
            // Get parent info if parent_id exists
            if ($data['parent_id'] != 0) {
                $parent = $this->tree->findById($data['parent_id']);
                if (! $parent) {
                    $v->errors()->add('parent_id', 'parent not found');
                }
            }
            // Validate currency logic
            if (! empty($data['currency_id'])) {
                // If currency is specified, it must match parent's currency
                if ($parent && $parent->currency_id !== null && $parent->currency_id !== $data['currency_id']) {
                    $v->errors()->add('currency_id', 'currency invalid with parent');
                }
            }
            // Validate correct level based on parent
            $expectedLevel = $parent ? $parent->accounts_structure_id + 1 : 1;
            if ($data['accounts_structure_id'] != $expectedLevel) {
                $v->errors()->add('accounts_structure_id', 'the level dont match accounts structure');
            }
            // Validate serial number format based on level (dynamic from accounts_structure)
            $level = (int) $data['accounts_structure_id'];
            $serialNumberDight = (string) $data['serial_number_dight'];

            if (! $this->validateSerialNumberFormat($serialNumberDight, $level, $parent)) {
                $v->errors()->add('serial_number_dight', 'invalid serial number format');
                // Prepare data for creation
                // Convert dashed serial to plain digits to store in serial_number
            }

            // Ensure uniqueness of the current level segment under the same parent
            $digitsOnly = preg_replace('/[^0-9]/', '', $serialNumberDight);
            $levelStructForUnique = AccountsStructure::where('id', $level)->first();
            if ($levelStructForUnique) {
                $finalLen = (int) $levelStructForUnique->end_position;
                $candidateCode = str_pad($digitsOnly, $finalLen, '0', STR_PAD_LEFT);
                $parentIdForUnique = (int) ($data['parent_id'] ?? 0);
                $isUnique = Helper::isSerialUnique($parentIdForUnique, $level, $candidateCode);
                if (! $isUnique) {
                    $v->errors()->add('serial_number_dight', 'serial not unique under parent');
                }
            }
        });
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // Create the account
        return $this->tree->create($data);
    }

    /**
     * Validate serial number format based on accounts_structure dynamically.
     */
    private function validateSerialNumberFormat(string $serialWithDashes, int $level, $parent = null): bool
    {
        $levels = AccountsStructure::orderBy('id')->get(['id', 'field_length', 'end_position', 'start_position']);

        if ($levels->isEmpty()) {
            return false;
        }

        $inputParts = explode('-', $serialWithDashes);
        $providedDepth = count($inputParts);
        if ($providedDepth < 1) {
            return false;
        }
        // لو المستخدم بعت مستويات أكثر من المعرفة حالياً، هنكتفي بالتحقق حتى آخر مستوى معروف

        // Build pattern only for the first N parts we want to validate
        $validateDepth = min($providedDepth, $levels->count());
        $lengthsForDepth = $levels->take($validateDepth)->pluck('field_length')->map(fn ($l) => (int) $l)->toArray();
        $partsPatterns = array_map(fn ($len) => '\\d{'.$len.'}', $lengthsForDepth);
        // اسمح بأجزاء إضافية بعد الأجزاء المتحققة
        $dynamicPattern = '/^'.implode('-', $partsPatterns).'(?:-.*)?$/';
        if (! preg_match($dynamicPattern, $serialWithDashes)) {
            return false;
        }

        // Plain digits and expected total length for provided depth
        $digits = str_replace('-', '', $serialWithDashes);
        if (! ctype_digit($digits)) {
            return false;
        }
        $finalForDepth = (int) $levels->get($validateDepth - 1)->end_position;
        // بدلاً من الرفض عند قِصر الطول الإجمالي، نستخدم padding لليمين للأصفار للاختبارات الداخلية
        $digitsPadded = strlen($digits) < $finalForDepth
            ? str_pad($digits, $finalForDepth, '0', STR_PAD_RIGHT)
            : $digits;

        // Find structures for current level and parent level
        $levelStruct = AccountsStructure::where('id', $level)->first();
        if (! $levelStruct) {
            return false;
        }
        $initial = (int) $levelStruct->start_position;
        $length = (int) $levelStruct->field_length;
        $finalForLevel = (int) $levelStruct->end_position;

        // Parent prefix check (validate only up to parent's end)
        if ($parent) {
            $parentLevel = (int) $parent->accounts_structure_id;
            $parentStruct = AccountsStructure::where('id', $parentLevel)->first();
            if (! $parentStruct) {
                return false;
            }
            $parentFinal = (int) $parentStruct->end_position;
            // Provided depth must at least cover parent's end
            if ($finalForDepth < $parentFinal) {
                return false;
            }
            // Use parent's dashed serial to preserve internal zeros
            $parentDigitsFull = str_replace('-', '', (string) $parent->serial_number_dight);
            // Ensure we are using exactly parent's end_position length
            $parentDigits = substr($parentDigitsFull, 0, $parentFinal);
            if (substr($digitsPadded, 0, $parentFinal) !== $parentDigits) {
                return false;
            }
        } else {
            // No parent: everything before current segment must be zeros
            if ($initial > 1) {
                $before = substr($digitsPadded, 0, $initial - 1);
                if ($before !== str_repeat('0', $initial - 1)) {
                    return false;
                }
            }
        }

        // All digits AFTER the current segment up to CURRENT LEVEL end must be zeros
        // Ignore deeper segments beyond current level (allows passing full code)
        $afterPos = $initial + $length - 1; // 1-based
        $afterLenWithinLevel = $finalForLevel - $afterPos;
        if ($afterLenWithinLevel > 0) {
            $afterWithinLevel = substr($digitsPadded, $afterPos, $afterLenWithinLevel);
            if ($afterWithinLevel !== str_repeat('0', $afterLenWithinLevel)) {
                return false;
            }
        }

        // For levels > 1 and has parent, all digits BETWEEN parent's end and current segment start must be zeros
        if ($parent) {
            $parentLevel = (int) $parent->accounts_structure_id;
            $parentStruct = AccountsStructure::where('id', $parentLevel)->first();
            if (! $parentStruct) {
                return false;
            }
            $parentFinal = (int) $parentStruct->end_position;

            if ($initial - 1 > $parentFinal) {
                $betweenLen = ($initial - 1) - $parentFinal;
                $between = substr($digitsPadded, $parentFinal, $betweenLen);
                if ($between !== str_repeat('0', $betweenLen)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Generate next serial number based on level and parent
     *
     * @param  object|null  $parent
     */
    public function generateNextSerial(int $level, $parent = null): string
    {
        // استخدم المنطق الديناميكي في Helper لتوليد الرقم التالي لأي عدد من المستويات
        $parentId = $parent ? (int) ($parent->id ?? 0) : 0;
        $result = \App\sys\Helper::generateNextSerial($parentId, $level);

        if (is_array($result) && isset($result['serial_number_dight'])) {
            return (string) $result['serial_number_dight'];
        }

        // fallback آمن في حال عدم رجوع صيغة متوقعة
        $struct = AccountsStructure::where('id', $level)->first();
        if ($struct) {
            $totalLen = (int) $struct->end_position;
            $zeros = str_repeat('0', max(0, $totalLen));

            return \App\sys\Helper::formatSerial($zeros, $level);
        }

        return '0';
    }

    /**
     * Get base serial parts from parent
     *
     * @param  object|null  $parent
     */
    private function getBaseSerialFromParent($parent): array
    {
        if (! $parent) {
            return ['1', '0', '00', '00', '000'];
        }

        $parentSerial = $parent->serial_number_dight;

        return explode('-', $parentSerial);
    }

    /**
     * Get next second part for level 2
     */
    private function getNextSecondPart(string $firstPart): string
    {
        $maxSecondPart = $this->tree->getMaxSecondPart($firstPart);

        return str_pad($maxSecondPart + 1, 1, '0', STR_PAD_LEFT);
    }

    /**
     * Get next third part for level 3
     */
    private function getNextThirdPart(string $firstPart, string $secondPart): string
    {
        $maxThirdPart = $this->tree->getMaxThirdPart($firstPart, $secondPart);

        return str_pad($maxThirdPart + 1, 2, '0', STR_PAD_LEFT);
    }

    /**
     * Get next fourth part for level 4
     */
    private function getNextFourthPart(string $firstPart, string $secondPart, string $thirdPart): string
    {
        $maxFourthPart = $this->tree->getMaxFourthPart($firstPart, $secondPart, $thirdPart);

        return str_pad($maxFourthPart + 1, 2, '0', STR_PAD_LEFT);
    }

    /**
     * Get next fifth part for level 5
     */
    private function getNextFifthPart(string $firstPart, string $secondPart, string $thirdPart, string $fourthPart): string
    {
        $maxFifthPart = $this->tree->getMaxFifthPart($firstPart, $secondPart, $thirdPart, $fourthPart);

        return str_pad($maxFifthPart + 1, 3, '0', STR_PAD_LEFT);
    }

    public function getAll()
    {
        return $this->tree->getAll();
    }
}
