<?php

namespace App\sys\Services\Accounting;

use App\sys\Repository\Accounting\PaytypeRepository;
use App\sys\Repository\Accounting\UnpostedCollectionsRepository;
use App\sys\Repository\General\CurrencyRepository;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Repository\Profile\TravelerRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UnpostedCollectionsServices extends Services
{
    private $unpostedRepo;

    private $profileAccount;

    private $payType;

    private $currency;

    private $invoces;

    public function __construct()
    {
        $this->unpostedRepo = new UnpostedCollectionsRepository;
        $this->profileAccount = new TravelerRepository;
        $this->payType = new PayTypeRepository;
        $this->currency = new CurrencyRepository;
        $this->invoces = new InvoiceServicesRepository;
    }

    public function add($data)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],
            'pay_type_id' => ['required', 'integer', 'exists:pay_type,id'],
            'date' => ['required', 'date'],
            'next_date' => ['nullable', 'date', 'after:today'],
            'amount' => ['required', 'numeric', 'min:0'],
            'reference' => ['nullable'],
            'details' => ['nullable'],
            'customer_name' => ['nullable'],
        ];
        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
            $amount = $this->unpostedRepo->getAmountForCurrnecy($data['profile_id'], $data['currency_id']);
            $finalAmount = $amountprofile - $amount;
            if ($finalAmount < $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than currency amount');
            }
        });
        $pay = $this->payType->getpayTypeAccountingByCurrency($data['pay_type_id'], $data['currency_id']);
        if (! empty($pay)) {
            if ($pay->max_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than max limit');
            }
            if ($pay->min_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than min limit');
            }
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        if (! empty($pay)) {
            $data['fee_amount'] = $data['amount'] * ($pay->percentage / 100);
            $data['transaction_amount'] = $data['amount'] - $data['fee_amount'];
        } else {
            $data['transaction_amount'] = $data['amount'];
            $data['fee_amount'] = 0;
        }
        // get Currnecy rate
        $data['currency_rate'] = $this->currency->findByIdOrFail($data['currency_id'])?->exchange_rate ?? null;

        return $this->unpostedRepo->add($data);
    }

    public function update($data)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],
            'pay_type_id' => ['required', 'integer', 'exists:pay_type,id'],
            'date' => ['required', 'date'],
            'next_date' => ['nullable', 'date', 'after:today'],
            'amount' => ['required', 'numeric', 'min:0'],
            'reference' => ['nullable'],
            'details' => ['nullable'],
            'customer_name' => ['nullable'],
        ];
        $validator = Validator::make($data, $rules);
        $validator->after(function ($validator) use ($data) {
            $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
            $amount = $this->unpostedRepo->getAmountForCurrnecyIgonrId($data['id'], $data['profile_id'], $data['currency_id']);
            $finalAmount = $amountprofile - $amount;
            if ($finalAmount < $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than currency amount');
            }
        });
        $pay = $this->payType->getpayTypeAccountingByCurrency($data['pay_type_id'], $data['currency_id']);
        if (! empty($pay)) {
            if ($pay->max_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than max limit');
            }

            if ($pay->min_amount > $data['amount']) {
                $validator->errors()->add('amount', 'amount greater than min limit');
            }
        }
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        if (! empty($pay)) {
            $data['fee_amount'] = $data['amount'] * ($pay->percentage / 100);
            $data['transaction_amount'] = $data['amount'] - $data['fee_amount'];
        } else {
            $data['transaction_amount'] = $data['amount'];
            $data['fee_amount'] = 0;
        }
        $data['currency_rate'] = $this->currency->findByIdOrFail($data['currency_id'])?->exchange_rate ?? null;

        return $this->unpostedRepo->updated($data);
    }

    public function getByProfile($profile_id)
    {
        return $this->unpostedRepo->getByProfileId($profile_id);
    }

    public function getAmount($data)
    {
        $rules = [
            'id' => ['nullable', 'integer', 'exists:unposted_collections,id'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id', 'exists:pr_profile_travelers,profile_id'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id',
                Rule::exists('pr_profile_travelers', 'currency_id')
                    ->where(fn ($query) => $query->where('profile_id', $data['profile_id'])),
            ],

        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $amountprofile = $this->profileAccount->getAmountByCurrnecy($data['profile_id'], $data['currency_id']);
        $amount = $data['id'] != null ? $this->unpostedRepo->getAmountForCurrnecyIgonrId($data['id'], $data['profile_id'], $data['currency_id']) : $this->unpostedRepo->getAmountForCurrnecy($data['profile_id'], $data['currency_id']);

        return [

            'remaining_amount' => (float) number_format($amountprofile - $amount, 3, '.', ''),
            'profile_amount' => (float) number_format($amountprofile, 3, '.', ''),
            'current_amount' => (float) number_format($amount, 3, '.', ''),
        ];
    }

    public function summary($profile_id)
    {
        // التكلفه
        $cost = $this->profileAccount->getByprofile($profile_id);
        // currency
        $currency = $cost->pluck('currency_id')->toArray();
        // المحصل
        $collector = $this->unpostedRepo->getPostedByCurrnecy($profile_id);
        // المدفوع final_currency_id
        $inovesIn = $this->invoces->summeryinCurrency($profile_id, $currency);
        $data = [];
        foreach ($currency as $key) {
            $currency_name = $this->currency->findByIdOrFail($key);
            $costBycurrency = $cost->where('currency_id', $key)->first();
            $collectorBycurrency = $collector->where('currency_id', $key)->first();
            $purchase = $inovesIn->where('final_currency_id', $key)->first();
            $data[] = [
                'currency_id' => $key,
                'currency_name' => $currency_name->name ?? null,
                'cost' => $costBycurrency->total ?? 0,
                'collector' => $collectorBycurrency->total_amount ?? 0,
                'purchase_price' => ($purchase->purchase_price ?? 0) + ($purchase->grand_total ?? 0),
                'remaining' => ($costBycurrency->total ?? 0) - ($collectorBycurrency->total_amount ?? 0),
                'profit' => ($costBycurrency->total ?? 0) - ($purchase->purchase_price ?? 0) + ($purchase->grand_total ?? 0),
                'currency_type' => 'mine_profile',
            ];
        }
        $inovesOut = $this->invoces->summeryinCurrencyNotIn($profile_id, $currency);
        if (! empty($inovesOut)) {
            foreach ($inovesOut as $key) {
                $data[] = [
                    'currency_id' => (int) $key->final_currency_id,
                    'currency_name' => $key->currency_name,
                    'cost' => 0,
                    'collector' => 0,
                    'purchase_price' => ($key->purchase_price ?? 0) + ($key->grand_total ?? 0),
                    'remaining' => 0,
                    'profit' => 0,
                    'currency_type' => 'out_profile',
                ];
            }
        }

        return $data;
    }

    public function selectedCurrency($profile_id)
    {
        return $this->profileAccount->selectedCurencyToProfile($profile_id);
    }
}
