<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CountryRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CountryService extends Services
{
    protected CountryRepository $countryRepository;

    public function __construct(CountryRepository $countryRepository)
    {
        $this->countryRepository = $countryRepository;
    }

    public function getPaginatedCountries()
    {
        return $this->countryRepository->getPaginated();
    }

    public function getCountryById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:countries,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->countryRepository->findByIdOrFail($id);
    }

    public function createCountry($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:255', 'unique:countries,name'],
            'code' => ['required', 'string', 'max:3'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $country = $this->countryRepository->create($request);
        $this->syncTranslations($country);

        return $country;
    }

    public function updateCountry($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:countries,id'],
            'name' => ['sometimes', 'string', 'max:255', 'unique:countries,name,'.$request['id']],
            'code' => ['sometimes', 'string', 'max:3'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        $country = $this->countryRepository->findByIdOrFail($request['id']);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->countryRepository->update($country, $request);
    }

    public function deleteCountry(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:countries,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $country = $this->countryRepository->findByIdOrFail($id);

        return $this->countryRepository->delete($country);
    }

    public function del(array $ids)
    {
        return $this->countryRepository->del($ids);
    }

    public function getActive()
    {
        return $this->countryRepository->getActive();
    }



    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:countries,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $country = $this->countryRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($country, true);
    }

    public function getAllChanges($country)
    {
        return $this->countryRepository->getAllChanges($country);
    }
}
