<?php

namespace App\sys\Services\General;

use App\sys\Repository\General\CurrencyRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class CurrencyService extends Services
{
    protected CurrencyRepository $currencyRepository;

    public function __construct(CurrencyRepository $currencyRepository)
    {
        $this->currencyRepository = $currencyRepository;
    }

    public function getPaginatedCurrencies()
    {
        return $this->currencyRepository->getPaginated();
    }

    public function getCurrencyById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:currencies,id'],
        ];

        $validator = Validator::make(['id' => $id], $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->currencyRepository->findByIdOrFail($id);
    }

    public function createCurrency($request)
    {
        $rules = [
            'name' => ['required', 'string', 'max:100', 'unique:currencies,name'],
            'symbol' => ['required', 'string', 'max:5'],
            'exchange_rate' => ['required', 'numeric'],
            'code' => ['required', 'string', 'max:10'],
            'active' => ['required', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $currency = $this->currencyRepository->create($request);
        $this->syncTranslations($currency);

        return $currency;
    }

    public function updateCurrency($request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:currencies,id'],
            'name' => ['sometimes', 'string', 'max:50', 'unique:currencies,name,'.$request['id']],
            'symbol' => ['sometimes', 'string', 'max:5'],
            'exchange_rate' => ['sometimes', 'numeric'],
            'code' => ['sometimes', 'string', 'max:10'],
            'active' => ['sometimes', 'in:0,1'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $currency = $this->currencyRepository->findByIdOrFail($request['id']);

        return $this->currencyRepository->update($currency, $request);
    }

    public function getActive()
    {
        return $this->currencyRepository->getActive();
    }

    public function getByIdWithTranslation($id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:currencies,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        $currency = $this->currencyRepository->getByIdWithTranslation($id);

        return $this->syncTranslations($currency, true);
    }

    public function del(array $ids)
    {
        return $this->currencyRepository->del($ids);
    }

    public function getAllChanges($currency)
    {
        return $this->currencyRepository->getAllChanges($currency);
    }

    public function getDefault()
    {
        return $this->currencyRepository->getDefault();
    }
}
