<?php

namespace App\sys\Services\Invoice;

use App\Models\Profile\ProfileTraveler;
use App\sys\Repository\Invoice\InvoiceServicesRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class InvoiceServicesService extends Services
{
    protected InvoiceServicesRepository $invoiceServicesRepository;

    public function __construct(InvoiceServicesRepository $invoiceServicesRepository)
    {
        $this->invoiceServicesRepository = $invoiceServicesRepository;
    }

    public function create(array $request)
    {

        // Validate service type
        if (! isset($request['travel_tourism_type'])) {
            $this->setError(['travel_tourism_type' => ['Service type is required']]);

            return false;
        }

        $serviceType = $request['travel_tourism_type'];

        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            $this->setError(['travel_tourism_type' => ['Unsupported service type']]);

            return false;
        }

        // Get validation rules for the service type
        $rules = InvoiceServicesTypeConfig::getValidationRules($serviceType);
        // Merge type-specific handler rules
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $rules = array_merge($rules, $handler->getCreateRules($request));
        }
        // Add common required fields
        $rules['travel_tourism_type'] = 'required|string|in:'.implode(',', InvoiceServicesTypeConfig::getSupportedTypes());

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        // Delegate to type handler to prepare data
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $prepared = $handler->prepareForCreate($request);
            if ($prepared === false) {
                $this->setError(['unknown' => ['unknown error in type handler']]);

                return false;
            }
            if (is_array($prepared) && array_key_exists('errors', $prepared)) {
                $this->setError($prepared['errors']);

                return false;
            }
            if (is_array($prepared)) {
                $request = $prepared;
            }
        }

        $created = $this->invoiceServicesRepository->create($request);
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $handler->afterCreate($created, $request);
        }

        return $created;
    }

    public function update(array $request)
    {
        // Require target id
        if (! array_key_exists('id', $request) || is_null($request['id'])) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail((int) $request['id']);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        // Get service type from existing item or request
        $serviceType = $request['travel_tourism_type'] ?? $item->travel_tourism_type;

        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            $this->setError(['travel_tourism_type' => ['Unsupported service type']]);

            return false;
        }

        // Get validation rules for the service type (only for provided fields)
        $rules = [];
        $providedFields = array_keys($request);

        foreach ($providedFields as $field) {
            $typeRules = InvoiceServicesTypeConfig::getValidationRules($serviceType);
            if (isset($typeRules[$field])) {
                $rules[$field] = $typeRules[$field];
            }
        }

        // Add common validation for provided fields
        if (isset($request['travel_tourism_type'])) {
            $rules['travel_tourism_type'] = 'required|string|in:'.implode(',', InvoiceServicesTypeConfig::getSupportedTypes());
        }
        if (isset($request['type'])) {
            $rules['type'] = 'required|string|in:credit,debit';
        }

        // Merge type-specific update rules
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $rules = array_merge($rules, $handler->getUpdateRules($request));
        }

        if (! empty($rules)) {
            $validator = Validator::make($request, $rules);

            if ($validator->fails()) {
                $this->setError($validator->errors());

                return false;
            }
        }

        // Delegate to type handler to prepare update data
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $prepared = $handler->prepareForUpdate($request, $item);
            if ($prepared === false) {
                $this->setError(['unknown' => ['unknown error in type handler']]);

                return false;
            }
            if (is_array($prepared) && array_key_exists('errors', $prepared)) {
                $this->setError($prepared['errors']);

                return false;
            }
            if (is_array($prepared)) {
                $request = $prepared;
            }
        }

        $updated = $this->invoiceServicesRepository->update($item, $request);
        if ($handler = InvoiceServicesTypeConfig::resolveHandler($serviceType)) {
            $handler->afterUpdate($updated, $request);
        }

        return $updated;
    }

    public function delete(int $id)
    {
        if (is_null($id)) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail($id);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        return $this->invoiceServicesRepository->delete($item);
    }

    public function deleteMany(array $ids)
    {
        $validator = \Illuminate\Support\Facades\Validator::make(['ids' => $ids], [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:invoice_services,id'],
        ]);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->invoiceServicesRepository->deleteManyWithRules($ids);
    }

    public function getById(int $id)
    {
        if (is_null($id)) {
            $this->setError(['id' => ['id is required']]);

            return false;
        }

        $item = $this->invoiceServicesRepository->findByIdOrFail($id);

        if (! $item) {
            $this->setError(['id' => ['Invoice service not found']]);

            return false;
        }

        return $item;
    }

    public function getByProfile(int $profileId, ?string $serviceType = null)
    {
        return $this->invoiceServicesRepository->getByProfile($profileId, $serviceType);
    }

    public function getByDaily(int $daily, ?string $serviceType = null)
    {
        return $this->invoiceServicesRepository->getByDaily($daily, $serviceType);
    }

    public function getList(array $filters = [])
    {
        return $this->invoiceServicesRepository->getList($filters);
    }

    public function getAllChanges(int $id)
    {
        $item = $this->getById($id);
        if (! $item) {
            return false;
        }

        return $this->invoiceServicesRepository->getAllChanges($item);
    }

    /**
     * Get validation rules for a specific service type
     */
    public function getValidationRulesForType(string $serviceType): array
    {
        if (! InvoiceServicesTypeConfig::isSupportedType($serviceType)) {
            return [];
        }

        return InvoiceServicesTypeConfig::getValidationRules($serviceType);
    }

    /**
     * Get supported service types
     */
    public function getSupportedServiceTypes(): array
    {
        return InvoiceServicesTypeConfig::getSupportedTypes();
    }

    /**
     * Ensure nationality exists under profile travelers and requested counts do not exceed stored counts
     */
    private function validateTravelerCounts(int $profileId, int $nationalityId, int $adultsRequested, int $childrenRequested)
    {
        $totals = ProfileTraveler::query()
            ->where('profile_id', $profileId)
            ->where('nationality_id', $nationalityId)
            ->selectRaw("SUM(CASE WHEN type='adult' THEN count ELSE 0 END) as adults_total")
            ->selectRaw("SUM(CASE WHEN type='child' THEN count ELSE 0 END) as children_total")
            ->first();

        if (! $totals || ((int) $totals->adults_total + (int) $totals->children_total) === 0) {
            return [
                'nationality_id' => ['لا يوجد مسافرون بهذه الجنسية مرتبطون بالبروفايل'],
            ];
        }

        $errors = [];
        if ($adultsRequested > (int) $totals->adults_total) {
            $errors['adults_count'][] = 'عدد الكبار المطلوب أكبر من المسجل في المسافرين';
        }
        if ($childrenRequested > (int) $totals->children_total) {
            $errors['children_count'][] = 'عدد الأطفال المطلوب أكبر من المسجل في المسافرين';
        }

        return empty($errors) ? true : $errors;
    }

    public function search()
    {
        return $this->invoiceServicesRepository->search();
    }
}
