<?php

namespace App\sys\Services\Invoice\Types;

use App\Models\General\Currency;
use App\Models\General\Service as GeneralService;
use App\Models\General\TaxRate;
use App\sys\Enums\ServicesType;

class TourGuideHandler implements InvoiceTypeHandlerInterface
{
    public function getCreateRules(array $request): array
    {
        return [
            'service_id' => ['required', 'integer', 'exists:services,id'],
            'travel_tourism_type' => ['required', 'in:'.ServicesType::TOUR_GUIDE->value],
            'daily_program_id' => ['required', 'integer', 'exists:pr_daily_programs,id'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'supplier_id' => ['required', 'integer', 'exists:su_supplier,id'],
            'guidance_place' => ['nullable', 'string', 'max:255'],
            'people_count' => ['required', 'integer', 'min:0'],
            'daily_cost' => ['required', 'numeric', 'min:0'],
            'tip_amount' => ['nullable', 'numeric', 'min:0'],
            'currency_id' => ['required', 'integer', 'exists:currencies,id'],
            'notes' => ['nullable', 'string', 'max:1000'],

            // computed/prohibited
            'currency_rate' => 'prohibited',
            'total_tips' => 'prohibited',
            'total_tax' => 'prohibited',
            'grand_total' => 'prohibited',

            // optional taxes array like accommodation service
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];
    }

    public function getUpdateRules(array $request): array
    {
        $rules = [
            'service_id' => ['sometimes', 'required', 'integer', 'exists:services,id'],
            'daily_program_id' => ['sometimes', 'required', 'integer', 'exists:pr_daily_programs,id'],
            'profile_id' => ['sometimes', 'required', 'integer', 'exists:pr_profile,id'],
            'supplier_id' => ['sometimes', 'required', 'integer', 'exists:su_supplier,id'],
            'guidance_place' => ['sometimes', 'nullable', 'string', 'max:255'],
            'people_count' => ['sometimes', 'required', 'integer', 'min:0'],
            'daily_cost' => ['sometimes', 'required', 'numeric', 'min:0'],
            'tip_amount' => ['sometimes', 'nullable', 'numeric', 'min:0'],
            'currency_id' => ['sometimes', 'required', 'integer', 'exists:currencies,id'],
            'notes' => ['sometimes', 'nullable', 'string', 'max:1000'],
            'tax_rate_id' => ['sometimes', 'array'],
            'tax_rate_id.*' => ['integer', 'exists:taxs_rate,id'],
        ];

        return $rules;
    }

    public function prepareForCreate(array $request): array|false
    {
        // Validate service type mapping
        $service = GeneralService::find($request['service_id']);
        if (! $service || $service->type !== ServicesType::TOUR_GUIDE->value) {
            return ['errors' => ['service_id' => ['service type must be tour_guide']]];
        }

        // Force guide_id null (tour_guide now uses only supplier_id)
        $request['guide_id'] = null;

        // Currency rate
        $currency = Currency::find($request['currency_id']);
        $request['currency_rate'] = $currency?->exchange_rate ?? 1;

        // Totals
        $people = (int) ($request['people_count'] ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? 0);
        $tip = (float) ($request['tip_amount'] ?? 0);

        $subtotal = $dailyCost;
        $totalTips = $people * $tip;

        // Taxes: sum of provided tax rates percentage
        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (is_array($taxRateIds) && ! empty($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        }

        $request['total_tips'] = $totalTips;
        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $totalTips + $totalTax;

        return $request;
    }

    public function prepareForUpdate(array $request, object $existing): array|false
    {
        // Optional validate service type if provided
        if (array_key_exists('service_id', $request)) {
            $service = GeneralService::find($request['service_id']);
            if (! $service || $service->type !== ServicesType::TOUR_GUIDE->value) {
                return ['errors' => ['service_id' => ['service type must be tour_guide']]];
            }
        }

        // Ensure guide remains null on updates
        if (array_key_exists('guide_id', $request)) {
            $request['guide_id'] = null;
        }

        // Currency rate refresh
        $currencyId = (int) ($request['currency_id'] ?? $existing->currency_id);
        if ($currencyId) {
            $currency = Currency::find($currencyId);
            $request['currency_rate'] = $currency?->exchange_rate ?? ($existing->currency_rate ?? 1);
        }

        $people = (int) ($request['people_count'] ?? $existing->people_count ?? 0);
        $dailyCost = (float) ($request['daily_cost'] ?? $existing->daily_cost ?? 0);
        $tip = (float) ($request['tip_amount'] ?? $existing->tip_amount ?? 0);

        $subtotal = $dailyCost;
        $totalTips = $people * $tip;

        $totalTax = 0.0;
        $taxRateIds = $request['tax_rate_id'] ?? null; // if not provided, keep old taxes recalculated from existing?
        if (is_array($taxRateIds)) {
            $rates = TaxRate::whereIn('id', $taxRateIds)->pluck('percentage');
            $sumPercent = (float) $rates->sum();
            $totalTax = ($subtotal * $sumPercent) / 100.0;
        } else {
            // fallback: keep existing total_tax if available
            $totalTax = (float) ($existing->total_tax ?? 0);
        }

        $request['total_tips'] = $totalTips;
        $request['total_tax'] = $totalTax;
        $request['grand_total'] = $subtotal + $totalTips + $totalTax;

        return $request;
    }

    public function afterCreate(object $invoiceService, array $request): void
    {
        $this->saveTaxes($invoiceService, $request);
    }

    public function afterUpdate(object $invoiceService, array $request): void
    {
        if (array_key_exists('tax_rate_id', $request)) {
            // re-save taxes if tax_rate_id provided
            $this->saveTaxes($invoiceService, $request, true);
        }
    }

    private function saveTaxes(object $invoiceService, array $request, bool $replace = false): void
    {
        $taxRateIds = $request['tax_rate_id'] ?? [];
        if (! is_array($taxRateIds)) {
            return;
        }

        if ($replace) {
            \App\Models\invoice\InvoiceServicesTax::where('invoice_services_id', $invoiceService->id)->delete();
        }

        $subtotal = (float) ($request['daily_cost'] ?? $invoiceService->daily_cost ?? 0);
        $rates = TaxRate::whereIn('id', $taxRateIds)->get(['id', 'percentage']);
        foreach ($rates as $rate) {
            $amount = ($subtotal * (float) $rate->percentage) / 100.0;
            \App\Models\invoice\InvoiceServicesTax::create([
                'invoice_services_id' => $invoiceService->id,
                'tax_rate_id' => $rate->id,
                'percentage' => (float) $rate->percentage,
                'tax_amount' => $amount,
            ]);
        }
    }
}
