<?php

namespace App\sys\Services\Profile;

use App\Models\Profile\AccommodationReservation;
use App\sys\Helper;
use App\sys\Repository\Profile\AccommodationReservationRepository;
use App\sys\Services;
use Illuminate\Support\Facades\Validator;

class AccommodationReservationService extends Services
{
    protected AccommodationReservationRepository $reservationRepository;

    public function __construct(AccommodationReservationRepository $reservationRepository)
    {
        $this->reservationRepository = $reservationRepository;
    }

    public function getPaginated(array $filters = [])
    {
        return $this->reservationRepository->getPaginated($filters);
    }

    public function getById(int $id)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['id' => $id], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->findByIdOrFail($id);
    }

    public function create(array $request)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'accommodation_id' => ['required', 'integer', 'exists:ac_accommodations,id'],
            'type' => ['required', 'in:local,international'],
            // 'status' => ['required', 'in:pending,confirmed,cancelled'],
            'confirmation_date' => ['nullable', 'date'],
            'confirmation_num' => ['nullable', 'string', 'max:100'],
            'supplier_id' => ['nullable', 'integer', 'exists:su_supplier,id'],
            'city_id' => ['required', 'integer', 'exists:cities,id'],
            'country_id' => ['required', 'integer', 'exists:countries,id'],
            'customer_note' => ['nullable', 'string'],
            'hotel_note' => ['nullable', 'string'],
            'reservation_num' => ['sometimes', 'nullable'],
            'reservation_link' => ['sometimes', 'nullable'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->create($request);
    }

    public function update(array $request)
    {
        $rules = [
            'id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
            'profile_id' => ['sometimes', 'integer', 'exists:pr_profile,id'],
            'accommodation_id' => ['sometimes', 'integer', 'exists:ac_accommodations,id'],
            'type' => ['sometimes', 'in:local,international'],
            'status' => ['sometimes', 'in:pending,confirmed,cancelled'],
            'confirmation_date' => ['sometimes', 'nullable', 'date'],
            'confirmation_num' => ['sometimes', 'nullable', 'string', 'max:100'],
            'supplier_id' => ['sometimes', 'integer', 'exists:su_supplier,id'],
            'city_id' => ['sometimes', 'integer', 'exists:cities,id'],
            'country_id' => ['sometimes', 'integer', 'exists:countries,id'],
            'customer_note' => ['sometimes', 'nullable', 'string'],
            'hotel_note' => ['sometimes', 'nullable', 'string'],
            'reservation_num' => ['sometimes', 'nullable'],
            'reservation_link' => ['sometimes', 'nullable'],
        ];

        $validator = Validator::make($request, $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $item = $this->reservationRepository->findByIdOrFail($request['id']);

        return $this->reservationRepository->update($item, $request);
    }

    public function del(array $ids)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['ids' => $ids], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->del($ids);
    }

    public function getByAccommodationType(string $type)
    {
        $rules = [
            'type' => ['required', 'in:hotel,cruise'],
        ];
        $validator = Validator::make(['type' => $type], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getByAccommodationType($type);
    }

    public function getByAccommodationTypeAndProfile(string $type, int $profileId)
    {
        $rules = [
            'type' => ['required', 'in:hotel,cruise'],
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['type' => $type, 'profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getByAccommodationTypeAndProfile($type, $profileId);
    }

    public function reservationAttachments(array $request)
    {
        if (isset($request['reservation_id']) && ! empty($request['reservation_id'])) {
            $request['attachable_type'] = (new AccommodationReservation)->getMorphClass();
            $request['attachable_id'] = (int) $request['reservation_id'];
        }

        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
            'attachable_type' => ['required', 'string'],
            'attachable_id' => ['required', 'integer'],
            'attachment' => ['required', 'string'],
            'name' => ['sometimes', 'nullable', 'string'],
        ];

        if (! empty($request['attachable_type']) && class_exists($request['attachable_type'])) {
            $className = $request['attachable_type'];
            $request['attachable_type'] = (new $className)->getMorphClass();
        }

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        $saveAttachment = Helper::saveFiles($request['attachment'], 'upload/reservation');

        if (! $saveAttachment['status']) {
            $this->setError(['attachment' => $saveAttachment['errors']]);

            return false;
        }
        $request['path'] = $saveAttachment['path'];

        // Default name to null if not provided
        if (! array_key_exists('name', $request)) {
            $request['name'] = null;
        }

        return $this->reservationRepository->uploadAttachments($request);
    }

    public function deleteAttachment($request)
    {
        $rules = [
            'ids' => ['required', 'array', 'min:1'],
            'ids.*' => ['required', 'integer', 'exists:pr_attachments,id'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }
        // Unlink files from disk first
        $attachments = $this->reservationRepository->findAttachmentsByIds($request['ids']);
        foreach ($attachments as $attachment) {
            $fullPath = public_path($attachment->path);
            if (file_exists($fullPath)) {
                @unlink($fullPath);
            }
        }

        return $this->reservationRepository->deleteAttachments($request['ids']);
    }

    public function getAttachmentsByProfileId(int $profileId)
    {
        $rules = [
            'profile_id' => ['required', 'integer', 'exists:pr_profile,id'],
        ];
        $validator = Validator::make(['profile_id' => $profileId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getAttachmentsByProfileId($profileId);
    }

    public function getAttachmentsByReservationId(int $reservationId)
    {
        $rules = [
            'reservation_id' => ['required', 'integer', 'exists:pr_accommodation_reservation,id'],
        ];
        $validator = Validator::make(['reservation_id' => $reservationId], $rules);
        if ($validator->fails()) {
            $this->setError($validator->errors());

            return false;
        }

        return $this->reservationRepository->getAttachmentsByReservationId($reservationId);
    }
}
